/*

	Library for writing cellular automaton rule definition files.

	Designed and implemented by John Walker in June of 1988.
	Extended for 8 bits of state per cell by John Walker in
	January of 1989.

	The process of compressing rule files uses dynamically
	allocated memory buffers (acquired by malloc()) which can, in
	the case of a completely random rule, consume as much as 64K
	of memory.  To allow for this case, this file and all programs
	which call it should be compiled for large memory model,
	allowing allocation of more than 64K of total data.  No memory
	allocation request asks for more than 256 bytes.

	If the program runs out of memory for compression buffers, the
	rule file will still be generated but it will not be optimally
	compressed.

	Every attempt has been made to make this file compatible with
	all C compilers.  If you run into trouble with your compiler,
        try removing the "#define COMPRESS" below.  This will generate
	much larger rule files, but it turns off most of the code that
	may confuse the odd C compiler.

*/

#include <stdio.h>
#include <ctype.h>

#define COMPRESS		   /* Generate compressed rule definitions */

/*  Rule loading instruction operation codes  */

#define RLUNCOMP 1		   /* 64K of uncompressed rule follows */
#define RLRUN	 2		   /* 3-256 byte run of value follows */
#define RLONEB	 3		   /* Single byte of specified value follows */
#define RLUNCS	 4		   /* Uncompressed string follows */
#define RLCOPYB  5		   /* Copy previously specified bank */
#define RLEND	 6		   /* End of rule definition */

#define RSHTEXT  64		   /* Horizontal texture specification */
#define RSVTEXT  65		   /* Vertical texture specification */
#define RSRAND	 66		   /* Random bit specification */
#define RSPAT	 67		   /* Request load of pattern */
#define RSPAL	 68		   /* Request load of palette */
#define RSRSEED  71		   /* Initial random seed */
#define RSOCODE  72		   /* Request load of user own code */

#define FALSE	 0
#define TRUE	 1

/*  Forward functions  */

void usage();
int cktext();
#ifdef COMPRESS
extern char *malloc();
char *alloc();
void pagerun();
#endif

/*	Main program.  This compiled file is linked with a another file
	which defines a function declared as:

	int jcrule(oldstate,	 nw, n	, ne,
				 w, self, e,
				 sw, s	, se
		   )
	int oldstate, nw, n, ne, w, self, e, sw, s, se;
	{
	    ...whatever...

	    return New_value;
	}

	This function is called 65536 times to evaluate each possible
	action of the single-plane cellular automaton based on the old
	value of the current cell from 0 to 255.  The function must
	return the resulting new value of the cell, from 0 to 255.  Only
	the low-order bit of the value returned is relevant to whether
        the cell will be considered "on" as seen by its neighbours.
	The high-order bits may be used to distinguish states by
	colour, or to remember additional state information for
	the central cell.

	The result of all possible state transitions is written as
	a rule definition file which may be loaded into the cellular
	automaton simulator for execution.

        For example, the following jcrule() function defines Conway's
	original game of Life.	It illustrates how straightforwardly
	the description of a rule translates into a jcrule() function
	to define it.  If you #define LIFE in this file, it will
	compile into a self-contained program which generates the rule
	file for Life.

*/

#ifdef LIFE

/*  John Horton Conway's original game of Life.  */

static int jcrule(oldstate,	nw, n  , ne,
				w, self, e,
				sw, s  , se
		  )
int oldstate, nw, n, ne, w, self, e, sw, s, se;
{
	int count;

	count = nw + n + ne + w + e + sw + s + se;

	if (count == 2) 	   /* If two neighbours... */
	   return self; 	   /*	...unchanged.	   */
	if (count == 3) 	   /* If exactly three...  */
	   return 1;		   /*	...cell is born.   */
	return 0;		   /* Otherwise die of loneliness or
				      overpopulation. */
}
#endif

#ifdef PARITY1

/*  Fredkin's parity rule for 1 bit per cell, 1D.  */

static int jcrule(oldstate,	l4, l3, l2, l1,
				     self,
				r1, r2, r3, r4
		  )
int oldstate, l4, l3, l2, l1, self, r1, r2, r3, r4;
{
	extern int worldtype;

	worldtype = 3;
	return l1 ^ r1;
}
#endif

#ifdef PARITY2

/*  Fredkin's parity rule for 2 bits per cell, 1D.  */

static int jcrule(oldstate,	l21, l20, l11, l10,
				       self,
				r11, r10, r21, r20
		  )
int oldstate, l21, l20, l11, l10, self, r11, r10, r21, r20;
{
	extern int worldtype;

	worldtype = 5;
	return ((l11 << 1) | l10) ^ ((r11 << 1) | r10);
}
#endif

#ifdef PARITY4

/*  Fredkin's parity rule for 4 bits per cell, 1D.  */

static int jcrule(oldstate,	l3, l2, l1, l0,
				     self,
				r3, r2, r1, r0
		  )
int oldstate, l3, l2, l1, l0, self, r3, r2, r1, r0;
{
	extern int worldtype;

	worldtype = 9;
	return ((((((l3 << 1) | l2) << 1) | l1) << 1) | l0) ^
	       ((((((r3 << 1) | r2) << 1) | r1) << 1) | r0);
}
#endif

int worldtype = 1;		   /* World type:  0 = 2D open plane
						   1 = 2D closed torus
						   2 = 1D line	  8 neighbours
						   3 = 1D circle  8 neighbours
						   4 = 1D line	  4 neighbours
						   5 = 1D circle  4 neighbours
						   8 = 1D line	  2 neighbours
						   9 = 1D circle  2 neighbours
						  10 = 2D semitotalistic 8 sum
						  11 = 2D semitotalistic 4 sum
						  12 = Own code plane
						  13 = Own code torus
						  -1 = unspecified */

int randdens = 0;		   /* Random number plane enable
				      and density specification:
					0     = disable random plane
				      1 : 255 = likelihood of random firing
				       -1     = unspecified */

int auxplane = 0;		   /* Auxiliary plane configuration:
				       -1 = Unspecified
					0 = Defined by rule or unused
					1 = Temporal phase
					2 = Horizontal texture
					4 = Vertical texture
				  hence 6 = Checkerboard texture */

int texthb = -1,		   /* Horizontal texture low bit */
    texthn = -1;		   /* Horizontal texture bit count */

int textvb = -1,		   /* Vertical texture low bit */
    textvn = -1;		   /* Vertical texture bit count */

int randb = -1, 		   /* Random low bit */
    randn = -1; 		   /* Number of random bits */

int rseedb = -1,		   /* Initial random seed low bit */
    rseedn = -1,		   /* Number of initial random seed bits */
    rseedp = 255;		   /* Initial random seed density (0-255) */

char patreq[258] = "";             /* Requested pattern file */
char palreq[258] = "";             /* Requested palette file */
char ocodereq[258] = "";           /* Requested user own code file */

/*  Main program  */

main(argc, argv)
int argc;
char *argv[];
{
	int i, j, k, l, m;
	char *op, opt;
	char filename[256];
	FILE *fp;
#ifdef COMPRESS
	char *pagebuf[256];	   /* Compression page buffers */
	char thispage[256];	   /* Current page before compression */
#endif

	fp = NULL;
	for (i = 1; i < argc; i++) {
	   op = argv[i];
           if (*op == '-') {
	      opt = *(++op);
	      if (islower(opt))
		 opt = toupper(opt);
	      switch (opt) {

                 case '?':
                 case 'U':
		    usage();
		    return;
	      }
	   } else {
	      if (fp) {
                 printf("Duplicate rule file name specification.\n");
		 return;
	      }
	      strcpy(filename, op);
              if (strchr(filename, '.') == NULL)
                 strcat(filename, ".jc");
              fp = fopen(filename, "wb");
	      if (fp == NULL) {
                 fprintf(stderr, "Cannot open rule definition file %s\n",
		    filename);
		 usage();
		 return;
	      }
	   }
	}

	if (fp == NULL) {
           fprintf(stderr, "No rule definition file name specified.\n");
	   usage();
	   return;
	}

#define bit(x)	 ((j >> (x)) & 1)

#ifndef COMPRESS
	putc(RLUNCOMP, fp);	   /* Indicate uncompressed rule follows */
#endif
	for (i = 0; i < 256; i++) {
	   for (j = 0; j < 256; j++) {

	      /* Now this is really sleazy; we use a different bit
		 order for one- and two-dimensional rules, but we
                 don't know whether the rule is one- or two-dimensional
                 until we've executed the rule the first time.
		 But...guess what?  The first time we execute the
		 rule function, all the inputs are zero, so the
                 bit order doesn't matter!  So...we just switch the bit
		 order on subsequent calls after the rule has
		 specified the world type.  The same applies to all the
		 other flavours of rule generation; the zero case is always
		 identical. */

	      if (worldtype == 12 || worldtype == 13) {
                 /* It's a user own-code rule */
		 m = (i << 8) | j;
		 l = jcrule(m, 0, 0, 0, 0, 0, 0, 0, 0, 0);
		 if (l < 0 || l > 255) {
                    fprintf(stderr, "Value returned by jcrule function, %d\n",
		       l);
                    fprintf(stderr, " when called as jcrule(%d,\n", m);
                    fprintf(stderr, "         0, 0, 0, 0, 0, 0, 0, 0, 0)\n");
                    fprintf(stderr, " is undefined.  Must be 0 <= value <= 255.\n");
		    l = 0;
		 }
	      } else if (worldtype == 10 || worldtype == 11) {
                 /* It's an 8 bit semitotalistic rule */
		 m = (i << 8) | j;
		 l = jcrule((m >> 11) & 0x1F,
			    m & 0x7FF, 0, 0,
			    0, i & 1, 0, 0, 0, 0);
		 if (l < 0 || l > 255) {
                    fprintf(stderr, "Value returned by jcrule function, %d\n",
		       l);
                    fprintf(stderr, " when called as jcrule(%d, %d,\n",
			  (m >> 11) & 0x1F, m & 0x7FF);
                    fprintf(stderr, "                       0, 0, 0, %d, 0,\n",
			  i & 1);
                    fprintf(stderr, "                       0, 0, 0)\n");
                    fprintf(stderr, " is undefined.  Must be 0 <= value <= 255.\n");
		    l = 0;
		 }
	      } else if ((worldtype >= 0) && ((worldtype & 0xE) != 0)) {
                 /* It's a 1D rule */
		 l = jcrule(((i << 1) | ((i >> 7) & 1)) & 0xFF,
				     bit(7), bit(6), bit(5),
				     bit(4), !!(i & 0x80), bit(3),
				     bit(2), bit(1), bit(0));
		 if (l < 0 || l > 255) {
                    fprintf(stderr, "Value returned by jcrule function, %d\n",
		       l);
                    fprintf(stderr, " when called as jcrule(%d,   %d, %d, %d\n",
			  ((i << 1) | ((i >> 7) & 1)) & 0xFF,
			  bit(7), bit(6), bit(5));
                    fprintf(stderr, "                             %d, %d, %d\n",
			  bit(4), !!(i & 0x80) , bit(3));
                    fprintf(stderr, "                             %d, %d, %d)\n",
			  bit(2), bit(1), bit(0));
                    fprintf(stderr, " is undefined.  Must be 0 <= value <= 255.\n");
		    l = 0;
		 }
	      } else {
                 /* It's a two-dimensional rule */
		 l = jcrule(((i << 1) | ((i >> 7) & 1)) & 0xFF,
				     bit(7), bit(6), bit(5),
				     bit(1), !!(i & 0x80), bit(0),
				     bit(4), bit(3), bit(2));
		 if (l < 0 || l > 255) {
                    fprintf(stderr, "Value returned by jcrule function, %d\n",
		       l);
                    fprintf(stderr, " when called as jcrule(%d,   %d, %d, %d\n",
			  ((i << 1) | ((i >> 7) & 1)) & 0xFF,
			  bit(7), bit(6), bit(5));
                    fprintf(stderr, "                             %d, %d, %d\n",
			  bit(1), !!(i & 0x80) , bit(0));
                    fprintf(stderr, "                             %d, %d, %d)\n",
			  bit(4), bit(3), bit(2));
                    fprintf(stderr, " is undefined.  Must be 0 <= value <= 255.\n");
		    l = 0;
		 }
	      }
	      /* Transform to internal bit ordering */
	      l = ((l >> 1) & 0x7F) | ((l & 1) << 7);
#ifdef COMPRESS
	      thispage[j] = l;
#else
	      putc(l, fp);
#endif
	   }

#ifdef COMPRESS

           /* If we're compressing, try to find an economical way to
	      output the current page.

              Step 1:  See if it's the same as a previously output
		       page.  If so, just copy it.  */

	   for (k = 0; k < i; k++) {
	      if (pagebuf[k] != NULL) {
		 for (l = 0; l < 256; l++) {
		    if (thispage[l] != pagebuf[k][l]) {
		       l = -1;
		       break;
		    }
		 }

		 if (l > 0) {
		    /* Aha!  This page is a duplicate.	Emit a copy page
		       instruction and set the page table pointer cell
                       for this page to NULL so subsequent pages don't
		       look at it for a prototype. */
		    pagebuf[i] = NULL;
		    putc(RLCOPYB, fp); /* Copy page instruction */
		    putc(k, fp);   /* Page index to copy */
		    k = -1;	   /* Indicate duplicate found */
		    break;
		 }
	      }
	   }

	   if (k >= 0) {

              /* Step 2.  Well, we didn't find a duplicate of this
			  page.  Output the page as a run-length
			  encoded stream, and save a copy of it in
                          case it's a duplicate of another page we
			  encounter later.  */

	      pagebuf[i] = alloc(256);
	      if (pagebuf[i] != NULL) {
		 for (k = 0; k < 256; k++)
		    pagebuf[i][k] = thispage[k];
	      }

	      pagerun(thispage, fp);

	   }
#endif

	}

	/* Output requested horizontal texture */

        if (cktext("horizontal texture", "texth", texthb, texthn)) {
	   putc(RSHTEXT, fp);
	   putc(texthb, fp);
	   putc(texthn, fp);
	}

	/* Output requested vertical texture */

        if (cktext("vertical texture", "textv", textvb, textvn)) {
	   putc(RSVTEXT, fp);
	   putc(textvb, fp);
	   putc(textvn, fp);
	}

	/* Output requested random input */

        if (cktext("random input", "rand", randb, randn)) {
	   putc(RSRAND, fp);
	   putc(randb, fp);
	   putc(randn, fp);
	}

	/* Output requested initial random seed */

        if (cktext("initial random seed", "rseed", rseedb, rseedn)) {
	   putc(RSRSEED, fp);
	   putc(rseedb, fp);
	   putc(rseedn, fp);
	   putc(rseedp, fp);
	}

	/* Output requested pattern */

	if (strlen(patreq) != 0) {
	   putc(RSPAT, fp);
	   putc(k = strlen(patreq) + 1, fp);
	   for (i = 0; i < k; i++) {
	      putc(patreq[i], fp);
	   }
	}

	/* Output requested palette file */

	if (strlen(palreq) != 0) {
	   putc(RSPAL, fp);
	   putc(k = strlen(palreq) + 1, fp);
	   for (i = 0; i < k; i++) {
	      putc(palreq[i], fp);
	   }
	}

	/* Output requested user own code file */

	if (strlen(ocodereq) != 0) {
	   putc(RSOCODE, fp);
	   putc(k = strlen(ocodereq) + 1, fp);
	   for (i = 0; i < k; i++) {
	      putc(ocodereq[i], fp);
	   }
	}


	putc(RLEND, fp);	   /* Write end of rule instruction */
	putc(worldtype, fp);
	putc(randdens, fp); 
	putc(auxplane, fp);
	fclose(fp);
}

/*  CKTEXT  --	Check texture specification  */

static int cktext(which, prefix, hb, hn)
char *which, *prefix;
int hb, hn;
{
	if (hn == -1 && hb == -1)
	   return FALSE;

	if (hn != -1 && hb == -1) {
	   fprintf(stderr,
              "Bad %s: %sn set and %sb unspecified.\n",
	       which, prefix, prefix);
	   return FALSE;
	}
	if (hb != -1 && hn == -1) {
	   fprintf(stderr,
              "Bad %s: %sb set and %sn unspecified.\n",
	      which, prefix, prefix);
	   return FALSE;
	}
	if (hn != -1) {
	   if (hn < 1 || hn > 7) {
	      fprintf(stderr,
                 "Bad %s: %sn out of range.\n", which, prefix);
	      return FALSE;
	   } else {
	      if (hb < 0 || hb > 7) {
		 fprintf(stderr,
                    "Bad %s: %sb out of range.\n", which, prefix);
		 return FALSE;
	      } else {
		 if ((hb + hn) > 8) {
		    fprintf(stderr,
                       "Bad %s: %sb+%sn > 8.\n", which, prefix, prefix);
		    return FALSE;
		 } else {
		    return TRUE;
		 }
	      }
	   }
	}
}

/*  USAGE  --  Print how to call information.  */

static void usage()
{
        fprintf(stderr, "\nUsage: jc_rule_maker rulefile[.jc]\n");
        fprintf(stderr, "  Options:\n");
        fprintf(stderr, "              -U  Print this message\n");
}

#ifdef COMPRESS

/*  PAGERUN  --  Output page as run-length compressed instruction
		 stream.  */

static void pagerun(buf, fp)
char *buf;
FILE *fp;
{
	char c, cn;
	char *cp, *cpr;
	int ll, rl, llr, cpi, cpri;
#define MINRUN	 3

	ll = 256;		   /* Initialise length left */
	cp = buf;
	cpi = 0;

	/* Loop until whole buffer is disposed of. */

	while (ll > 0) {

	   /* Search for run starting with current byte.  */

	   c = *cp;
	   cpr = cp + 1;
	   rl = 1;
	   llr = ll - 1;
	   while (llr > 0 && c == *cpr) {
	      llr--;
	      cpr++;
	      rl++;
	   }

	   /* If we found a run long enough to bother with, output
	      it. */

	   if (rl >= MINRUN) {
	      putc(RLRUN, fp);	   /* Output run instruction */
	      putc(rl - 1, fp);    /* Output run length */
	      putc(c, fp);	   /* Output contents of run */
	      cp += rl; 	   /* Advance past run */
	      cpi += rl;	   /* Increment page table index */
	      ll -= rl; 	   /* Update length left to output */
	   } else {

	      /* The current byte does not begin a worthwhile run.
		 Scan forward and determine the length of the
		 stream of bytes that precedes the next run. */

	      llr = 0;
	      for (cpr = cp + 1, cpri = cpi + 1;
		   cpri <= (cpi + (ll - MINRUN)); cpr++, cpri++) {
		 cn = *cpr;
		 for (rl = 1; rl < (MINRUN + 1); rl++) {
		    if (cn != cpr[rl]) {
		       rl = -1;
		       break;
		    }
		 }
		 if (rl >= 0) {
		    llr = 1;
		    break;
		 }
	      }

	      /* Output the next incompressible stream. */

	      if (llr) {
		 rl = cpr - cp;
	      } else {
		 rl = ll;
	      }

	      if (rl == 1) {
		 putc(RLONEB, fp);
		 putc(c, fp);
	      } else {
		 putc(RLUNCS, fp);
		 putc(rl - 1, fp);
		 for (llr = 0; llr < rl; llr++) {
		    c = cp[llr];
		    putc(c, fp);
		 }
	      }
	      cp += rl;
	      cpi += rl;
	      ll -= rl;
	   }
	}
}

/*  ALLOC  --  Allocate storage and diagnose out of memory conditions. */

static char *alloc(nb)
int nb;
{
	char *cp;
	static int allocfail = FALSE;

	if (allocfail)
	   return NULL;
	cp = malloc(nb);
	if (cp == NULL) {
	   allocfail = TRUE;
	   fprintf(stderr,
              "\nInsufficient memory to generate compressed rule;\n");
	   fprintf(stderr,
            "  rule file will be larger than necessary.\n");
	}
	return cp;
}
#endif
