/*

        Handy definitions for writing rules

*/

/*  Define sum of neighbour external state bits in various
    configurations. */

#define SUM_4    (n + w + e + s)
#define SUM_5    (n + w + self + e + s)
#define SUM_8    (nw + n + ne + w + e + sw + s + se)
#define SUM_9    (nw + n + ne + w + self + e + sw + s + se)

/*  Canned header for rule definition functions with standard argument
    names.  */

#define RULEDEF() int jcrule(oldstate,nw,n,ne,w,self,e,sw,s,se) \
        int oldstate,nw,n,ne,w,self,e,sw,s,se;

/*  Bit and bit mask generation functions.  */

/* Bit set for plane P */
#define BIT(p)   (1 << (p))

/* Mask for N contiguous bits with low order bit in plane P.  Note how
   this definition craftily generates masks of zero when a zero bit
   field is specified. */
#define BITMASK(p,n) (BIT((p) + (n)) - BIT((p)))

/* Test specific bit set in oldstate. */
#define BITSET(p)   ((oldstate & BIT(p)) != 0)

/* Extract field of contiguous bits from oldstate. */
#define BITFIELD(p,n) ((oldstate >> (p)) & BITMASK(0,(n)))

/* Place value in specified bit field. */
#define BF(v,p)  ((v) << (p))

/*  Horizontal and vertical phase extraction functions.  Assume
    that the user has defined:

           HPPlane     Horizontal phase plane (LSB)  0 - 7
           HPNbits     Number of bits of horizontal phase (1 - (8 - HPPlane))
           VPPlane     Vertical phase plane (LSB)  0 - 7
           VPNbits     Number of bits of vertical phase (1 - (8 - VPPlane))
           TPPlane     Temporal phase plane (LSB) 0 - 7
           TPNbits     Number of bits of temporal phase (1 - (8 - TPPlane))

    Then:

           HPHASE      Horizontal phase index
           VPHASE      Vertical phase index
           HVPHASE     Composite phase: the result of concatenating
                       VPHASE with HPHASE.
           TPHASE      Temporal phase

           TPUPD(x)    Apply to the result returned by the rule
                       definition function to increment the temporal
                       phase field for the next generation.

    These definitions assume that the texture bits are supplied
    in "oldstate", per standard rule definitions.
*/

#define HPHASE   ((oldstate >> (HPPlane)) & BITMASK(0, (HPNbits)))
#define VPHASE   ((oldstate >> (VPPlane)) & BITMASK(0, (VPNbits)))
#define HVPHASE  (((VPHASE) << (HPNbits)) | (HPHASE))
#define TPHASE   ((oldstate >> (TPPlane)) & BITMASK(0, (TPNbits)))

#define TPUPD(x) (((x) & (~BITMASK(TPPlane,TPNbits))) | \
                  (((TPHASE + 1) & BITMASK(0,(TPNbits))) << TPPlane))

/*          Definitions of one-dimensional neighbours

    The following macros generate expressions that compute the values
    of the neighbours for one-dimensional rules, based upon the
    standard rule function argument names declared by RULEDEF().
    Three sets of neighbour declarations are provided, choose the
    correct set based upon your choice of the tradeoff between the
    number of visible neighbours and the number of visible bits of
    neighbour state per cell, as expressed in the setting of
    worldtype.  The following table presents the options.

 Worldtype   Neighbours  Bits                 Neighbour Names

   2 or 3         8        1     N8L4 N8L3 N8L2 N8L1 self N8R1 N8R2 N8R3 N8R4
   4 or 5         4        2               N4L2 N4L1 self N4R1 N4R2
   8 or 9         2        4                    N2L1 self N2R1

    As with a two dimensional rule, you should use "self" for the current
    cell if you're interested only in the low-order bit of the cell and
    "oldstate" if you want to examine all 8 bits.  All one dimensional
    modes provide the standard 8 bits of local state per cell.

*/

/*  Neighbours for 8 neighbours, 1 bit visible from each  */
#define N8L4     nw
#define N8L3     n
#define N8L2     ne
#define N8L1     w
#define N8R1     e
#define N8R2     sw
#define N8R3     s
#define N8R4     se

/*  Neighbours for 4 neighbours, 2 bits visible from each  */
#define N4L2     ((nw << 1) | n)
#define N4L1     ((ne << 1) | w)
#define N4R1     ((e << 1) | sw)
#define N4R2     ((s << 1) | se)

/*  Neighbours for 2 neighbours, 4 bits visible from each  */
#define N2L1     ((((((nw << 1) | n) << 1) | ne) << 1) | w)
#define N2R1     ((((((e << 1) | sw) << 1) | s) << 1) | se)

/*  Declarations of global rule mode variables.  */

extern int worldtype, randdens, auxplane, texthb, texthn,
           textvb, textvn, randb, randn, rseedb, rseedn, rseedp;
extern char patreq[], palreq[], ocodereq[];
