/*

	Diffusion of a lattice gas

	This rule implements the Margolus rule for simulating an
	HPP-gas of particles bouncing off each other.  No external
	noise is used.	Particles are regarded as travelling
	horizontally or vertically.  We set up a lattice of position
	values that looks like this:

		 0 1 0 1 ..
		 2 3 2 3 ..
		 0 1 0 1 ..
		 2 3 2 3 ..
		 : : : :

	This lattice is alternately chunked into:

	    A blocks: 0 1   and   B blocks: 3 2
		      2 3		    1 0

	and each cell is swapped with is diagonally opposite cell,
	except when there is a collision, if that is a block has form:

	   1 0	 or 0 1
	   0 1	    1 0.

	When there is a collision the block is rotated one notch CCW.

	We use the eight bits of state as follows:

	Bit   #0 is the machine visible bit for update
	Bit   #1 is used for the gas
	Bit   #2 is the wall
	Bit   #3 is the touch wall in my neighborhood bit
	Bits  #4 and #5 hold a position number between 0 & 3
	Bit   #6 controls the check wall/do gas cycle
	Bit   #7 controls the A/B lattice cycle
*/

#include "jcrule.h"

#define HPPlane  4		   /* Horizontal phase plane */
#define HPNbits  1		   /* Horizontal phase plane count */
#define VPPlane  5		   /* Vertical phase plane */
#define VPNbits  1		   /* Vertical phase plane count */
#define TPPlane  6		   /* Temporal phase plane */
#define TPNbits  2		   /* Temporal phase plane count */

RULEDEF()
{
	static int firstime = 1;
	int touchwall, newtouchwall, wall, gas, newgas, newself;

	if (firstime) {

	   /* Specify modes on first invocation of rule function.

	      We set a horizontal pattern of alternate 0s and 1s in bit 4
	      and a vertical pattern of alternate 0s and 1s in bit 5.
	      This produces a pattern that goes 0 1 0 1 ..
						2 3 2 3 ..
						0 1 0 1 ..
						2 3 2 3 ..
						: : : :     */
	   firstime = 0;
	   texthb = HPPlane;
	   texthn = HPNbits;
	   textvb = VPPlane;
	   textvn = VPNbits;

	   /* The perfume.cac palette only shows bit 1 (hex mask 2). */

           strcpy(palreq, "perfume");

	   /* The starting Perfume pattern is wall around a glob of gas. */

           strcpy(patreq, "perfume");
	}

	touchwall = BITSET(3);		   /* Touching a wall */
	wall = BITSET(2);		   /* This is a wall */
	gas = BITSET(1);		   /* Gas particle here */

	switch (TPHASE) {

	   /* In both cycle0 / LatticeA and cycle1 / LatticeB do
	      if collision then newself = CW else newself = OPP */

	   case 0:

	      /* Set touch wall if any neighbor is on

		 Block has form 0 1
				2 3 */

	      switch (HVPHASE) {
		 case 0:
		    newtouchwall = (self + e + se + s) != 0;
		    break;

		 case 1:
		    newtouchwall = (self + s + sw + w) != 0;
		    break;

		 case 2:
		    newtouchwall = (self + n + ne + e) != 0;
		    break;

		 case 3:
		    newtouchwall = (self + w + nw + n) != 0;
		    break;
	       }
	       newgas = newself = gas;
	       break;

	   case 1:

	   /* Move gas unless collision or touching a wall

	      Block has form 0 1
			     2 3 */

	      switch (HVPHASE) {
		 case 0:
		    newgas = (((self == se) && (e == s)) || touchwall) ?
		       e : se;
		    break;

		 case 1:
		    newgas = (((self == sw) && (w == s)) || touchwall) ?
		       s : sw;
		    break;

		 case 2:
		    newgas = (((self == ne) && (e == n)) || touchwall) ?
		       n : ne;
		    break;

		 case 3:
		    newgas = (((self == nw) && (w == n)) || touchwall) ?
		       w : nw;
		    break;
	       }
	       newtouchwall = touchwall;
	       newself = wall;
	       break;

	   case 2:

	     /* Set touch wall if any neighbor is on

		Block has form 3 2
			       1 0 */

	      switch (HVPHASE) {
		 case 0:
		    newtouchwall = (self + w + nw + n) != 0;
		    break;

		 case 1:
		    newtouchwall = (self + n + ne + e) != 0;
		    break;

		 case 2:
		    newtouchwall = (self + s + sw + w) != 0;
		    break;

		 case 3:
		    newtouchwall = (self + e + se + s) != 0;
		    break;
	      }
	      newgas = newself = gas;
	      break;

	   case 3:

	   /* If collision newself = CW else newself = OPP

	      Block has form 3 2
			     1 0 */

	      switch (HVPHASE) {
		 case 0:
		    newgas = (((self == nw) && (w == n)) || touchwall) ?
		       w : nw;
		    break;

		 case 1:
		    newgas = (((self == ne) && (e == n)) || touchwall) ?
		       n : ne;
		    break;

		 case 2:
		    newgas = (((self == sw) && (w == s)) || touchwall) ?
		       s : sw;
		    break;

		 case 3:
		    newgas = (((self == se) && (e == s)) || touchwall) ?
		       e : se;
		    break;
	      }
	      newtouchwall = touchwall;
	      newself = wall;
	      break;
	}

	return	 /* The return value is all jammed together in one line
                    because some C compilers don't like arguments to
		    #define functions that span multiple lines. */
 TPUPD(BF(HVPHASE,HPPlane)|BF(newtouchwall,3)|BF(wall,2)|BF(newgas,1)|newself);
}
