{

	Pascal library for writing cellular automaton rule definition files.

	Designed and implemented by John Walker in November of 1988.
	Extended for 8 bits of local state per cell by John Walker
	in January of 1989.

	This unit is linked with a another file which defines a function
	declared as:

	function jcrule(oldstate,     nw, n  , ne,
				      w, self, e,
				      sw, s  , se : integer
		   ) : integer;
	begin
	    ...whatever...

	    jcrule := New_value
	end;

	This function is called 65536 times to evaluate each possible
	action of the single-plane cellular automaton based on the old
	value of the current cell from 0 to 255.  The function must
	return the resulting new value of the cell, from 0 to 255.  Only
	the low-order bit of the value returned is relevant to whether
        the cell will be considered "on" as seen by its neighbours.
	The high-order bits may be used to distinguish states by
	colour, or to remember a additional state information for
	the central cell.

	The result of all possible state transitions is written as
	a rule definition file which may be loaded into the cellular
	automaton simulator for execution.
}

unit jcmake;

interface

type

    RuleFunc = function(oldstate, nw, n   , ne,
				  w , self, e ,
				  sw, e   , sw : integer) : integer;

const

    worldtype : integer = 1;	   {  World type:  0 = 2D open plane
						   1 = 2D closed torus
						   2 = 1D line	  8 neighbours
						   3 = 1D circle  8 neighbours
						   4 = 1D line	  4 neighbours
						   5 = 1D circle  4 neighbours
						   8 = 1D line	  2 neighbours
						   9 = 1D circle  2 neighbours
						  10 = 2D semitotalistic 8 sum
						  11 = 2D semitotalistic 4 sum
						  12 = Own code plane
						  13 = Own code torus
						  -1 = unspecified  }

    randdens : integer = 0;	   {  Random number plane enable
				      and density specification:
					0     = disable random plane
				      1 : 255 = likelihood of random firing
				       -1     = unspecified  }

    auxplane : integer = 0;	   {  Auxiliary plane configuration:
				       -1 = Unspecified
					0 = Defined by rule or unused
					1 = Temporal phase
					2 = Horizontal texture
					4 = Vertical texture
				  hence 6 = Checkerboard texture  }

    texthb : integer = -1;	   { Horizontal texture low bit }
    texthn : integer = -1;	   { Horizontal texture bit count }

    textvb : integer = -1;	   { Vertical texture low bit }
    textvn : integer = -1;	   { Vertical texture bit count }

    randb : integer = -1;	   { Random low bit }
    randn : integer = -1;	   { Random bit count }

    rseedb : integer = -1;	   { Initial random seed low bit }
    rseedn : integer = -1;	   { Initial random seed bit count }
    rseedp : integer = 255;	   { Initial random seed density, 0-255 }

    patreq: string[255] = '';      { Requested pattern file }
    palreq: string[255] = '';      { Requested palette file }
    ocodereq: string[255] = '';    { Requested user own code file }

procedure genrule(jcrule: RuleFunc);

implementation

const

    IOBLEN   = 511;		   { I/O Buffer length - 1 }

{   Evaluate all possible cases of rule and write rule
    definition file.  }

procedure makerule(var f: file; jcrule: RuleFunc);
label
	3, 4;
const

	{ Compressed rule op-codes }

	RLUNCOMP = 1;		   { 64K of uncompressed rule follows }
	RLRUN	 = 2;		   { 3-256 byte run of value follows }
	RLONEB	 = 3;		   { Single byte of specified value follows }
	RLUNCS	 = 4;		   { Uncompressed string follows }
	RLCOPYB  = 5;		   { Copy previously specified bank }
	RLEND	 = 6;		   { End of rule definition }

	RSHTEXT  = 64;		   { Horizontal texture specification }
	RSVTEXT  = 65;		   { Vertical texture specification }
	RSRAND	 = 66;		   { Random bit specification }
	RSPAT	 = 67;		   { Request load of pattern }
	RSPAL	 = 68;		   { Request load of palette }
	RSRSEED  = 71;		   { Initial random seed }
	RSOCODE  = 72;		   { Request load of user own code }

type
	pagebuf = array[0..255] of char;
var
	i, j, k, l, m, x: integer;
	wasmatch: Boolean;
	c: char;
	sa: array[0..IOBLEN] of char;
	cpage: pagebuf;
	paget: array[0..255] of ^pagebuf;

function bit(x : integer) : integer;   { Extract bit from J }
begin
	bit := (j shr x) and 1
end;

function bit0 : integer;	       { Extract bit 0 from I }
begin
	bit0 := (i shr 7) and 1
end;

procedure putbyte(b: integer);	       { Put byte in buffer, write if full }
begin
	if x > IOBLEN then begin
	   BlockWrite(f, sa, 1);
	   x := 0;
	end;
	sa[x] := char(b);
	x := x + 1
end;

{	CKTEXT	--  Check texture bit numbers within range. }

function cktext(which, prefix: string; hb, hn: integer) : Boolean;
begin
	if (hn = -1) and (hb = -1) then
	   cktext := FALSE
	else begin

	   if (hn <> -1) and (hb = -1) then begin
              Writeln('Bad ', which, ': ', prefix, 'n set and ',
                 prefix, 'b unspecified.');
	      cktext := FALSE
	   end else begin
	      if (hb <> -1) and (hn = -1) then begin
                 Writeln('Bad ', which, ': ', prefix, 'b set and ',
                    prefix, 'n unspecified.');
		 cktext := FALSE
	      end else begin
		 if (hn <> -1) then begin
		    if (hn < 1) or (hn > 7) then begin
                       Writeln('Bad ', which, ': ', prefix,
                          'n out of range.');
		       cktext := FALSE
		    end else begin
		       if (hb < 0) or (hb > 7) then begin
                          Writeln('Bad ', which, ': ', prefix,
                             'b out of range.');
			  cktext := FALSE
		       end else begin
			  if (hb + hn) > 8 then begin
                             Writeln('Bad ', which, ': ', prefix, 'b+',
                                prefix, 'n > 8.');
			     cktext := FALSE
			  end else begin
			     cktext := TRUE
			  end
		       end
		    end
		 end
	      end
	   end
	end
end;

procedure pagerun;
label
	1, 2;
const
	MINRUN = 3;
var
	cp, cpr, ll, rl, llr: integer;
	fr, ulf: Boolean;
	c, cn: char;
begin
	ll := 256;		   { Length left }
	cp := 0;

	{ Loop until entire buffer is disposed of. }

	while ll > 0 do begin

	   { Search for run starting at current byte. }

	   c := cpage[cp];
	   cpr := cp + 1;
	   rl := 1;
	   llr := ll - 1;
	   while (llr > 0) and (c = cpage[cpr]) do begin
	      llr := llr - 1;
	      cpr := cpr + 1;
	      rl := rl + 1
	   end;

	   { If we found a run long enough to bother with, output it. }

	   if rl >= MINRUN then begin
	      putbyte(RLRUN);
	      putbyte(rl - 1);
	      putbyte(integer(c));
	      cp := cp + rl;
	      ll := ll - rl
	   end else begin

	      { The current byte does not begin a worthwhile run.
		Scan forward and determine the length of the stream
		of bytes that precedes the next run. }

	      ulf := false;
	      for cpr := cp + 1 to (cp + (ll - MINRUN)) do begin
		 cn := cpage[cpr];
		 fr := true;
		 for rl := 1 to MINRUN do begin
		    if cn <> cpage[cpr + rl] then begin
		       fr := false;
		       goto 1
		    end
		 end;
1:		 if fr then begin
		    ulf := true;
		    goto 2
		 end
	      end;

	      { Output the next incompressible stream. }

2:	      if ulf then
		 rl := cpr - cp
	      else
		 rl := ll;

	      if rl = 1 then begin
		 putbyte(RLONEB);
		 putbyte(integer(c))
	      end else begin
		 putbyte(RLUNCS);
		 putbyte(rl - 1);
		 for llr := 0 to rl - 1 do
		    putbyte(integer(cpage[cp + llr]))
	      end;

	      cp := cp + rl;
	      ll := ll - rl

	   end
	end
end;

begin
	x := 0;
	for i := 0 to 255 do begin
	   for j := 0 to 255 do begin

	      { If the world type is specified as one-dimensional,
		we must use a different ordering of the neighbours.
		Note that this means that a one-dimensional rule
		definition MUST set worldtype properly before calling
		genrule. }

	      if ((worldtype = 12) or (worldtype = 13)) then begin

		 { Generation for user-defined own code rules }

		 m := (i shl 8) or j;
		 l := jcrule(m, 0, 0, 0, 0, 0, 0, 0, 0, 0);
		 if (l < 0) or (l > 255) then begin
                    Writeln('Value returned by jcrule function, ', l);
                    Writeln(' when called as jcrule(', m,
                        ', 0, 0, 0, 0, 0, 0, 0, 0, 0);');
                    Writeln(' is undefined.  Must be 0 <= value <= 255.');
		    l := 0
		 end 
	      end else if ((worldtype = 10) or (worldtype = 11)) then begin

		 { Generation for semitotalistic rule }

		 m := (i shl 8) or j;
		 l := jcrule((m shr 11) AND $1F,
			     m AND $7FF, 0, 0,
			     0, i AND 1, 0, 0, 0, 0);
		 if (l < 0) or (l > 255) then begin
                    Writeln('Value returned by jcrule function, ', l);
                    Writeln(' when called as jcrule(',
		       (m shr 11) AND $1F,
                        ',   ', m AND $7FF, ', ', 0, ', ', 0, ',');
                    Writeln('                            ',
                                0, ', ', i AND 1, ', ', 0, ',');
                    Writeln('                            ',
                                0, ', ', 0, ', ', 0, ');');
                    Writeln(' is undefined.  Must be 0 <= value <= 255.');
		    l := 0
		 end 
	      end else if ((worldtype >= 0) and
			   ((worldtype and $E) <> 0)) then begin

		 { Generation for one-dimensional rules }

		 l := jcrule(((i shl 1) or ((i shr 7) and 1)) and 255,
				      bit(7), bit(6), bit(5),
				      bit(4), bit0,   bit(3),
				      bit(2), bit(1), bit(0));
		 if (l < 0) or (l > 255) then begin
                    Writeln('Value returned by jcrule function, ', l);
                    Writeln(' when called as jcrule(',
		       ((i shl 1) or ((i shr 7) and 1)) and 255,
                        ',   ', bit(7), ', ', bit(6), ', ', bit(5), ',');
                    Writeln('                            ',
                                bit(4), ', ', bit0, ', ', bit(3), ',');
                    Writeln('                            ',
                                bit(2), ', ', bit(1), ', ', bit(0), ');');
                    Writeln(' is undefined.  Must be 0 <= value <= 255.');
		    l := 0
		 end 

	      end else begin

		 { Generation for two-dimensional rules }

		 l := jcrule(((i shl 1) or ((i shr 7) and 1)) and 255,
				      bit(7), bit(6), bit(5),
				      bit(1), bit0,   bit(0),
				      bit(4), bit(3), bit(2));
		 if (l < 0) or (l > 255) then begin
                    Writeln('Value returned by jcrule function, ', l);
                    Writeln(' when called as jcrule(',
		       ((i shl 1) or ((i shr 7) and 1)) and 255,
                        ',   ', bit(7), ', ', bit(6), ', ', bit(5), ',');
                    Writeln('                            ',
                                bit(1), ', ', bit0, ', ', bit(0), ',');
                    Writeln('                            ',
                                bit(4), ', ', bit(3), ', ', bit(2), ');');
                    Writeln(' is undefined.  Must be 0 <= value <= 255.');
		    l := 0
		 end 
	      end;

	      cpage[j] := char(((l shr 1) and 127) or ((l and 1) shl 7));
	   end;

	   { Check if this page duplicates a previously-output page.
	     If so, simply ditto that page. }

	   for k:= 0 to i - 1 do begin
	      if (paget[k] <> nil) then begin
		 wasmatch := true;
		 for l := 0 to 255 do begin
		    if cpage[l] <> paget[k]^[l] then begin
		       wasmatch := false;
		       goto 3
		    end
		 end;
3:		 if wasmatch then begin
		    paget[i] := nil;
		    putbyte(RLCOPYB);
		    putbyte(k);
		    goto 4
		 end
	      end
	   end;
	   New(paget[i]);
	   for k := 0 to 255 do
	      paget[i]^[k] := cpage[k];
	   pagerun;
4:	end;

	{ Output requested horizontal texture }

        if cktext('horizontal texture', 'texth', texthb, texthn) then begin
	   putbyte(RSHTEXT);
	   putbyte(texthb);
	   putbyte(texthn)
	end;

	{ Output requested vertical texture }

        if cktext('vertical texture', 'textv', textvb, textvn) then begin
	   putbyte(RSVTEXT);
	   putbyte(textvb);
	   putbyte(textvn)
	end;

	{ Output requested random input }

        if cktext('random input', 'rand', randb, randn) then begin
	   putbyte(RSRAND);
	   putbyte(randb);
	   putbyte(randn)
	end;

	{ Output requested initial random seed }

        if cktext('initial random seed', 'rseed', rseedb, rseedn) then begin
	   putbyte(RSRSEED);
	   putbyte(rseedb);
	   putbyte(rseedn);
	   putbyte(rseedp)
	end;

	{ Output requested pattern file }

	if Length(patreq) <> 0 then begin
	   putbyte(RSPAT);
	   putbyte(Length(patreq) + 1);
	   for i := 1 to Length(patreq) do
	      putbyte(integer(patreq[i]));
	   putbyte(0)
	end;

	{ Output requested palette file }

	if Length(palreq) <> 0 then begin
	   putbyte(RSPAL);
	   putbyte(Length(palreq) + 1);
	   for i := 1 to Length(palreq) do
	      putbyte(integer(palreq[i]));
	   putbyte(0)
	end;

	{ Output requested user own code file }

	if Length(ocodereq) <> 0 then begin
	   putbyte(RSOCODE);
	   putbyte(Length(ocodereq) + 1);
	   for i := 1 to Length(ocodereq) do
	      putbyte(integer(ocodereq[i]));
	   putbyte(0)
	end;

	putbyte(RLEND);
	putbyte(worldtype);
	putbyte(randdens);
	putbyte(auxplane);
	BlockWrite(f, sa, 1)
end;

{  GENRULE  --	Generate rule into specified file name.
		Obtains rule file name from the first argument
		on the command line, if specified, otherwise
		by asking the user for the file name. }

procedure genrule;
var
    filename, query: string;
    outfile: file;
    clarg: Boolean;
begin
     clarg := ParamCount >= 1;
     if clarg then
	filename := ParamStr(1)
     else begin
        Write('Rule file name: ');
	Readln(filename)
     end;
     if Pos('.', filename) = 0 then
        filename := filename + '.jc';
     Assign(outfile, filename);
     Rewrite(outfile, IOBLEN + 1);
     makerule(outfile, jcrule);
     Close(outfile);
     Writeln('Rule file ', filename, ' generated.');
     if not clarg then begin
        Write('Press Enter to continue: ');
	Readln(query)
     end
end;

end.
