/*

						Fractal forgery generator

		Originally	designed  and  implemented	in December of 1989 by
		John Walker as a stand-alone program for the  Sun  and	MS-DOS
		under  Turbo C.  Adapted in September of 1991 for use with Jef
        Poskanzer's raster toolkit.

		References cited in the comments are:

			Foley, J. D., and Van Dam, A., Fundamentals of Interactive
				Computer  Graphics,  Reading,  Massachusetts:  Addison
				Wesley, 1984.

			Peitgen, H.-O., and Saupe, D. eds., The Science Of Fractal
				Images, New York: Springer Verlag, 1988.

			Press, W. H., Flannery, B. P., Teukolsky, S. A., Vetterling,
				W. T., Numerical Recipes In C, New Rochelle: Cambridge
				University Press, 1988.

	Author:
			John Walker
			http://www.fourmilab.ch/

	Permission	to	use, copy, modify, and distribute this software and
	its documentation  for	any  purpose  and  without	fee  is  hereby
	granted,  without any conditions or restrictions.  This software is
    provided "as is" without express or implied warranty.

*/

#include "screensv.h"

#define NoExtremeCrescents
#define RandomParameters

#ifndef M_PI
#define M_PI	3.14159265358979323846
#endif
#ifndef M_E
#define M_E 	2.7182818284590452354
#endif

/* Definitions used to address real and imaginary parts in a two-dimensional
   array of complex numbers as stored by fourn(). */

#define Real(v, x, y)  v[1 + (((x) * meshsize) + (y)) * 2]
#define Imag(v, x, y)  v[2 + (((x) * meshsize) + (y)) * 2]

/* Co-ordinate indices within arrays. */

#define X	  0
#define Y	  1
#define Z	  2

/* Definition for obtaining random numbers. */

#define nrand 4 					  /* Gauss() sample count */
#define Cast(low, high) ((low) + (((high) - (low)) * ((random() & 0x7FFF) / arand)))

/*	Data types	*/

typedef int Boolean;
#define FALSE 0
#define TRUE 1

/*	Pixel access definitions  */

static LPBYTE pixelArray = NULL;		// Pixel array
static int pixelLineLength = 0;			// Pixel array line length
	
#define SET_PIXEL(y, x, r, g, b) { LPBYTE p = pixelArray + (((screenysize - 1) - (y)) * pixelLineLength) + ((x) * 3); \
								   *p++ = b; *p++ = g; *p = r; }

/*	This is only used for stars.  If any of the components being added exceeds
	255, this really should scale all the components equally to prevent a shift
	in the colour balance.  In reality, with practical star densities and
	intensities overflow almost never happens.  Even if it does, reasonable
	settings for star colours results in stars very close to white, so even if
	limitation of a single pixel to 255 won't shift the colour balance very
	much from the ideal value.  */

#define ADD_PIXEL(y, x, r, g, b) { LPBYTE p = pixelArray + (((screenysize - 1) - (y)) * pixelLineLength) + ((x) * 3); \
								   p[0] = (BYTE) min((p[0] + b), 255); p[1] = (BYTE) min((p[1] + g), 255); p[2] = (BYTE) min((p[2] + r), 255); }

#define CondFree(x)	if ((x) != NULL) { free(x); x = NULL; }

/*	Black body temperature to RGB mapping table.  We use this table, with
	entries for temperatures between BBODY_TEMP_MIN and BBODY_TEMP_MAX
	every BBODY_TEMP_RESOLUTION degrees Kelvin, to look up RGB colour
	components corresponding to that temperature.  This avoids having to do
	the entire integration of the black body spectrum for every star we
	plot.  */

static unsigned char *black_to_pixel = NULL;
#define BBODY_TEMP_RESOLUTION	100		// Temperature resolution in degrees Kelvin
#define BBODY_TEMP_MIN			2600	// Minimum temperature
#define	BBODY_TEMP_MAX			28000	// Maximum temperature
#define BBODY_RGB(temp)	(black_to_pixel + (3 * ((((int) (temp)) - BBODY_TEMP_MIN) / BBODY_TEMP_RESOLUTION)))

/*	Local variables  */

static double arand, gaussadd, gaussfac; /* Gaussian random parameters */
static double fracdim;				  /* Fractal dimension */
static double powscale; 			  /* Power law scaling exponent */
static int meshsize = 512;			  /* FFT mesh size */
static Boolean seedspec = FALSE;	  /* Did the user specify a seed ? */
static Boolean clouds = FALSE;		  /* Just generate clouds */
static Boolean hell = FALSE;		  /* Generate hellfire */
static Boolean stars = FALSE;		  /* Just generate stars */
static int screenxsize = 256;		  /* Screen X size */
static int screenysize = 256;		  /* Screen Y size */
static double inclangle, hourangle;   /* Star position relative to planet */
static Boolean inclspec = FALSE;	  /* No inclination specified yet */
static Boolean hourspec = FALSE;	  /* No hour specified yet */
static double icelevel; 			  /* Ice cap theshold */
static double glaciers; 			  /* Glacier level */
static int starfraction;			  /* Star fraction */
static int starcolour;				  /* Star colour saturation */
static int planetX = 100, planetY = 100; /* Planet position on screen */
static int planetSize = 350;		  /* Planet size in pixels */

int lastPlanetX = -1;				  // Last planet horizontal position
int lastPlanetSize = 0;				  // Last planet size

//	Precomputed Gaussian point spread function for stars

#define GAUSSIAN_MAX	10			  /* Maximum pixel spread for stars (one-sided) */
#define GAUSSIAN_SPREAD	1			  /* Width of PSF (smaller = wider) */
static double gaussian_psf[GAUSSIAN_MAX][GAUSSIAN_MAX];

//	Star cluster table

struct Star_cluster {
	int clusterpos[2];				  // Centre of cluster on screen
	int clusterdens;				  // Cluster density
	int clusterwid;					  // Cluster width
};

static int nclusters = 0;			  // Number of clusters
static struct Star_cluster clusters[MAX_STAR_CLUSTERS];	// Table of star clusters

/*		FOURN  --  Multi-dimensional fast Fourier transform

		Called with arguments:

		   data 	  A  one-dimensional  array  of  floats  (NOTE!!!	NOT
					  DOUBLES!!), indexed from one (NOTE!!!   NOT  ZERO!!),
					  containing  pairs of numbers representing the complex
					  valued samples.  The Fourier transformed results	are
					  returned in the same array.

		   nn		  An  array specifying the edge size in each dimension.
					  THIS ARRAY IS INDEXED FROM  ONE,	AND  ALL  THE  EDGE
					  SIZES MUST BE POWERS OF TWO!!!

		   ndim 	  Number of dimensions of FFT to perform.  Set to 2 for
					  two dimensional FFT.

		   isign	  If 1, a Fourier transform is done; if -1 the	inverse
					  transformation is performed.

        This  function  is essentially as given in Press et al., "Numerical
        Recipes In C", Section 12.11, pp.  467-470.
*/

static void fourn(float data[], int nn[], int ndim, int isign)
{
	register int i1, i2, i3;
	int i2rev, i3rev, ip1, ip2, ip3, ifp1, ifp2;
	int ibit, idim, k1, k2, n, nprev, nrem, ntot;
	float tempi, tempr;
	double theta, wi, wpi, wpr, wr, wtemp;
	static int tickeroo = 0;
#define TICKEROO_INTERVAL	3

#define SWAP(a,b) tempr=(a); (a) = (b); (b) = tempr

	Ticker("FFT");
	ntot = 1;
	for (idim = 1; idim <= ndim; idim++) {
		ntot *= nn[idim];
	}
	nprev = 1;
	for (idim = ndim; idim >= 1; idim--) {
		n = nn[idim];
		nrem = ntot / (n * nprev);
		ip1 = nprev << 1;
		ip2 = ip1 * n;
		ip3 = ip2 * nrem;
		i2rev = 1;
		for (i2 = 1; i2 <= ip2; i2 += ip1) {
			if (((++tickeroo) & TICKEROO_INTERVAL) == 0) {
				ticker();
				if (bailout()) {
					return;
				}
			}
			if (i2 < i2rev) {
				for (i1 = i2; i1 <= i2 + ip1 - 2; i1 += 2) {
					for (i3 = i1; i3 <= ip3; i3 += ip2) {
						i3rev = i2rev + i3 - i2;
						SWAP(data[i3], data[i3rev]);
						SWAP(data[i3 + 1], data[i3rev + 1]);
					}
				}
			}
			ibit = ip2 >> 1;
			while (ibit >= ip1 && i2rev > ibit) {
				i2rev -= ibit;
				ibit >>= 1;
			}
			i2rev += ibit;
		}
		ifp1 = ip1;
		while (ifp1 < ip2) {
			if (((++tickeroo) & TICKEROO_INTERVAL) == 0) {
				ticker();
				if (bailout()) {
					return;
				}
			}
			ifp2 = ifp1 << 1;
			theta = isign * (M_PI * 2) / (ifp2 / ip1);
			wtemp = sin(0.5 * theta);
			wpr = -2.0 * wtemp * wtemp;
			wpi = sin(theta);
			wr = 1.0;
			wi = 0.0;
			for (i3 = 1; i3 <= ifp1; i3 += ip1) {
				for (i1 = i3; i1 <= i3 + ip1 - 2; i1 += 2) {
					for (i2 = i1; i2 <= ip3; i2 += ifp2) {
						k1 = i2;
						k2 = k1 + ifp1;
						tempr = (float) (wr * data[k2] - wi * data[k2 + 1]);
						tempi = (float) (wr * data[k2 + 1] + wi * data[k2]);
						data[k2] = data[k1] - tempr;
						data[k2 + 1] = data[k1 + 1] - tempi;
						data[k1] += tempr;
						data[k1 + 1] += tempi;
					}
				}
				wr = (wtemp = wr) * wpr - wi * wpi + wr;
				wi = wi * wpr + wtemp * wpi + wi;
			}
			ifp1 = ifp2;
		}
		nprev *= n;
	}
#undef TICKEROO_INTERVAL
}
#undef SWAP

/*	INITGAUSS  --  Initialise random number generators.  As given in
				   Peitgen & Saupe, page 77. */

static void initgauss(void)
{
	int i, j;

	/* Range of random generator */
	arand = pow(2.0, 15.0) - 1.0;
	gaussadd = sqrt(3.0 * nrand);
	gaussfac = 2 * gaussadd / (nrand * arand);

	//	Initialise precomputed table of gaussian point spread function

	for (i = 0; i < GAUSSIAN_MAX; i++) {
		double si = i * GAUSSIAN_SPREAD;

		for (j = 0; j < GAUSSIAN_MAX; j++) {
			double sj = j * GAUSSIAN_SPREAD;
			gaussian_psf[i][j] = exp(-sqrt((si * si) + (sj * sj)));
		}
	}
}

/*	GAUSS  --  Return a Gaussian random number.  As given in Peitgen
			   & Saupe, page 77. */

static double gauss(void)
{
	int i;
	double sum = 0.0;

	for (i = 1; i <= nrand; i++) {
		sum += (random() & 0x7FFF);
	}
	return gaussfac * sum - gaussadd;
}

/*	SPECTRALSYNTH  --  Spectrally  synthesised	fractal  motion in two
					   dimensions.	This algorithm is given under  the
					   name   SpectralSynthesisFM2D  on  page  108	of
					   Peitgen & Saupe. */

static void spectralsynth(float **x, int n, double h)
{
	unsigned int bl;
	int i, j, i0, j0, nsize[3];
	double rad, phase, rcos, rsin;
	float *a;

	Ticker("Synth");
	bl = ((((unsigned long) n) * n) + 1) * 2 * sizeof(float);
	a = (float *) calloc(bl, 1);
	if (a == (float *) 0) {
		loadError = "Cannot allocate array for inverse Fourier transform";
		return;
	}
	*x = a;

	for (i = 0; i <= n / 2; i++) {
		for (j = 0; j <= n / 2; j++) {
			phase = 2 * M_PI * ((random() & 0x7FFF) / arand);
			if (i != 0 || j != 0) {
				rad = pow((double) (i * i + j * j), -(h + 1) / 2) * gauss();
			} else {
				rad = 0;
			}
			rcos = rad * cos(phase);
			rsin = rad * sin(phase);
			Real(a, i, j) = (float) rcos;
			Imag(a, i, j) = (float) rsin;
			i0 = (i == 0) ? 0 : n - i;
			j0 = (j == 0) ? 0 : n - j;
			Real(a, i0, j0) = (float) rcos;
			Imag(a, i0, j0) = (float) (- rsin);
		}
	}
	Imag(a, n / 2, 0) = 0;
	Imag(a, 0, n / 2) = 0;
	Imag(a, n / 2, n / 2) = 0;
	for (i = 1; i <= n / 2 - 1; i++) {
		for (j = 1; j <= n / 2 - 1; j++) {
			phase = 2 * M_PI * ((random() & 0x7FFF) / arand);
			rad = pow((double) (i * i + j * j), -(h + 1) / 2) * gauss();
			rcos = rad * cos(phase);
			rsin = rad * sin(phase);
			Real(a, i, n - j) = (float) rcos;
			Imag(a, i, n - j) = (float) rsin;
			Real(a, n - i, j) = (float) rcos;
			Imag(a, n - i, j) = (float) (- rsin);
		}
	}

	nsize[0] = 0;
	nsize[1] = nsize[2] = n;		  /* Dimension of frequency domain array */
	fourn(a, nsize, 2, -1); 		  /* Take inverse 2D Fourier transform */
}

/*	ETOILE	--	Set a pixel in the starry sky.	*/

static void etoile(int y, int x)
{
	int i, cenhance = 0;

	//	Calculate density enhancement due to star clusters

	for (i = 0; i < nclusters; i++) {
		double cdx = x - clusters[i].clusterpos[X],
			   cdy = y - clusters[i].clusterpos[Y];
		cenhance = 10 * ((int) (clusters[i].clusterdens * exp(-sqrt((cdx * cdx) + (cdy * cdy)) / clusters[i].clusterwid)));
	}

	if ((random() % 1000) < (starfraction + cenhance)) {
#define StarQuality 	0.5 		  /* Brightness distribution exponent */
#define StarIntensity	8			  /* Brightness scale factor */
#define StarTintExp 	0.5 		  /* Tint distribution exponent */
		double v = StarIntensity * pow(1 / (1 - Cast(0, 0.9999)),
									   (double) StarQuality),
			   temp,
			   r, g, b;
		int i, j, ir, ig, ib;

		if (v > 255) {
			v = 255;
		}

		/* We make a special case for star colour  of zero in order to
		   prevent	floating  point  roundoff  which  would  otherwise
		   result  in  more  than  256 star colours.  We can guarantee
		   that if you specify no star colour, you never get more than
		   256 shades in the image. */

		if (starcolour == 0) {
			ir = ig = ib = (int) v;
		} else {
			unsigned char *bbrgb;

			temp = 5500 + starcolour *
						  pow(1 / (1 - Cast(0, 0.9999)), StarTintExp) *
							  ((random() & 7) ? -1 : 1);
			/* Constrain temperature to a reasonable value: >= 2600K 
			   (S Cephei/R Andromedae), <= 28,000 (Spica). */
			temp = max(BBODY_TEMP_MIN, min(BBODY_TEMP_MAX, temp));
#ifdef FULL_BLACK_BODY_CALCULATION
			/*	This is for documentation only.  To avoid integrating
				over the entire black body spectrum for every star, we
				use the precomputed black body RGB table in the #else
				clause below.  */
			black_body_rgb(temp, &r, &g, &b);
#else
			bbrgb = BBODY_RGB(temp);
			r = *bbrgb++ / 255.0;
			g = *bbrgb++ / 255.0;
			b = *bbrgb / 255.0;
#endif
			ir = (int) (r * v + 0.499);
			ig = (int) (g * v + 0.499);
			ib = (int) (b * v + 0.499);
		}
#ifdef SINGLE_PIXEL_STARS
		SET_PIXEL(y, x, ir, ig, ib);
#else
		for (i = 0; i < GAUSSIAN_MAX; i++) {
			for (j = 0; j < GAUSSIAN_MAX; j++) {
				double psf = gaussian_psf[i][j];
				int gir = (int) (ir * psf), gig = (int) (ig * psf),
					gib = (int) (ib * psf);

				if (i == j && gir == 0 && gig == 0 && gib == 0) {
					//	Bail out when PSF yields zero contribution to pixel
					goto gdone;
				}
				if (gir > 0 || gig > 0 || gib > 0) {
					//	Use symmetry to reduce iterations through loop
					if ((y - i) >= 0 && (x - j) >= 0) {
						ADD_PIXEL(y - i, x - j, gir, gig, gib);
					}
					if ((y - i) >= 0 && (x + j) < screenxsize) {
						ADD_PIXEL(y - i, x + j, gir, gig, gib);
					}
					if ((y + i) < screenysize && (x - j) >= 0) {
						ADD_PIXEL(y + i, x - j, gir, gig, gib);
					}
					if ((y + i) < screenysize && (x + j) < screenxsize) {
						ADD_PIXEL(y + i, x + j, gir, gig, gib);
					}
				}
			}
		}
#endif
	}
gdone:;
}

/*	GENPLANET  --  Generate planet from elevation array.  */

static void genplanet(float *a, int n)
{
	int i, j;
	int imageX, imageY, imageHeight, imageWidth;
	unsigned char *cp = NULL, *ap;
	double *u = NULL, *u1;
	unsigned int *bxf, *bxc;

#define RGBQuant	255

#define Atthick 1.03				  /* Atmosphere thickness as a percentage
                                         of planet's diameter */
	double athfac = sqrt(Atthick * Atthick - 1.0);
	double sunvec[3];
	Boolean flipped = FALSE;
	double shang, siang;
	static int tickeroo = 0;
#define TICKEROO_INTERVAL	3

	/*	Unless we're generating clouds, initialise the image to
		all black and add the stars.  */

	if (!clouds) {
		Ticker("Stars");
		memset(pixelArray, 0, pixelLineLength * screenysize);
		for (i = 0; i < screenysize; i++) {
			if (((++tickeroo) & TICKEROO_INTERVAL) == 0) {
				ticker();
				if (bailout()) {
					goto tickerbail;
				}
			}
			for (j = 0; j < screenxsize; j++) {
				etoile(i, j);
			}
		}
	}
	if (stars) {
		return;
	}

	if (clouds) {
		Ticker("Clouds");
	} else {
		Ticker("Planet");
	}
	if (!stars) {
		u = (double *) malloc((unsigned int) (screenxsize * sizeof(double)));
		u1 = (double *) malloc((unsigned int) (screenxsize * sizeof(double)));
		bxf = (unsigned int *) malloc((unsigned int) screenxsize *
			  sizeof(unsigned int));
		bxc = (unsigned int *) malloc((unsigned int) screenxsize *
			  sizeof(unsigned int));

		if (u == NULL || u1 == NULL ||
			bxf == NULL || bxc == NULL) {
			loadError = "Cannot allocate interpolation tables";
			CondFree(u);
			CondFree(u1);
			CondFree(bxf);
			CondFree(bxc);
			return;
		}

		/* Compute incident light direction vector. */

#ifdef NoExtremeCrescents
		shang = hourspec ? hourangle : Cast(-(M_PI * 5) / 8, (M_PI * 5) / 8);
#else
		shang = hourspec ? hourangle : Cast(0, 2 * M_PI);
#endif
		siang = inclspec ? inclangle : Cast(-M_PI * 0.12, M_PI * 0.12);

		sunvec[X] = sin(shang) * cos(siang);
		sunvec[Y] = sin(siang);
		sunvec[Z] = cos(shang) * cos(siang);

		/* Allow only 25% of random pictures to be crescents */

		if (!hourspec && ((random() % 100) < 75)) {
			flipped = sunvec[Z] < 0 ? TRUE : FALSE;
			sunvec[Z] = abs(sunvec[Z]);
		}

/*
        pm_message("%s: -seed %d -dimension %.2f -power %.2f -mesh %d",
            clouds ? (hell ? " hell" : "clouds") : "planet",
			rseed, fracdim, powscale, meshsize);
		if (!clouds) {
			pm_message(
               "        -inclination %.0f -hour %d -ice %.2f -glaciers %.2f",
			   (siang * (180.0 / M_PI)),
			   (int) (((shang * (12.0 / M_PI)) + 12 +
				  (flipped ? 12 : 0)) + 0.5) % 24, icelevel, glaciers);
            pm_message("        -stars %d -saturation %d.",
				starfraction, starcolour);
		}
*/

		/* Prescale the grid points into intensities. */

		cp = (unsigned char *) malloc(n * n);
		if (cp == NULL) {
			loadError = "Insufficient memory for elevation table";
			goto tickerbail;			// Guarantee memory gets freed
		}
		ap = cp;
		for (i = 0; i < n; i++) {
			for (j = 0; j < n; j++) {
				*ap++ = (unsigned char) ((255.0 * (Real(a, i, j) + 1.0)) / 2.0);
			}
		}

		/* Fill the screen from the computed  intensity  grid  by  mapping
		   screen  points onto the grid, then calculating each pixel value
		   by bilinear interpolation from  the	surrounding  grid  points.
		   (N.b. the pictures would undoubtedly look better when generated
		   with small grids if	a  more  well-behaved  interpolation  were
		   used.)

		   Before  we get started, precompute the line-level interpolation
           parameters and store them in an array so we don't  have  to  do
		   this every time around the inner loop. */

#define UPRJ(a,size) ((a)/((size)-1.0))

		for (j = 0; j < screenxsize; j++) {
			double bx = (n - 1) * UPRJ(j, screenxsize);

			bxf[j] = (unsigned int) floor(bx);
			bxc[j] = bxf[j] + 1;
			u[j] = bx - bxf[j];
			u1[j] = 1 - u[j];
		}
	}

	imageY = 0;
	imageX = 0;
	imageHeight = screenysize;
	imageWidth = screenxsize;

	for (i = imageY; i < imageHeight; i++) {
		double t, t1, by, dy, dysq, sqomdysq, icet, svx, svy, svz,
			   azimuth;
		int byf, byc, lcos;

		if (((++tickeroo) & TICKEROO_INTERVAL) == 0) {
			ticker();
			if (bailout()) {
				goto tickerbail;
			}
		}
		if (!stars) {				  /* Skip all this setup if just stars */
			by = (n - 1) * UPRJ(clouds ? i : i - planetY, clouds ? screenysize : planetSize);
			dy = 2 * ((((clouds ? screenysize : planetSize) / 2) - (clouds ? i : i - planetY)) /
					((double) (clouds ? screenysize : planetSize)));
			dysq = dy * dy;
			sqomdysq = sqrt(1.0 - dysq);
			svx = sunvec[X];
			svy = sunvec[Y] * dy;
			svz = sunvec[Z] * sqomdysq;
			byf = (int) (floor(by) * n);
			byc = byf + n;
			t = by - floor(by);
			t1 = 1 - t;
		}

		if (clouds) {

			/* Render the FFT output as clouds. */

			for (j = imageX; j < imageWidth; j++) {
				double r = t1 * u1[j] * cp[byf + bxf[j]] +
						   t  * u1[j] * cp[byc + bxf[j]] +
						   t  * u[j]  * cp[byc + bxc[j]] +
						   t1 * u[j]  * cp[byf + bxc[j]];
				int w = (r > 127.0) ? ((int) (RGBQuant * ((r - 127.0) / 128.0))) :
						   0;

				if (hell) {
					SET_PIXEL(i, j, RGBQuant, w, max(0, (int) w - 128));
				} else {
					SET_PIXEL(i, j, w, w, RGBQuant);
				}
			}
		} else { // Planet

			/*	If we're within the vertical extent of the planet on
				the screen, loop to generate its horizontal span on
				this line.  */

			if (i >= planetY && i < (planetY + planetSize)) {
				int li = i - planetY;
				 
				azimuth = asin(((((double) li) / (planetSize - 1)) * 2) - 1);
				icet = pow(abs(sin(azimuth)), 1.0 / icelevel) - 0.5;
				lcos = (int) ((planetSize / 2) * abs(cos(azimuth)));
				for (j = (planetSize / 2) - lcos;
					 j <= (planetSize / 2) + lcos; j++) {
					int lj = j - ((planetSize / 2) - lcos);
					double r = t1 * u1[lj] * cp[byf + bxf[lj]] +
							   t  * u1[lj] * cp[byc + bxf[lj]] +
							   t  * u[lj]  * cp[byc + bxc[lj]] +
							   t1 * u[lj]  * cp[byf + bxc[lj]],
						   ice;
					int ir, ig, ib;
					static unsigned char pgnd[][3] = {
					   {206, 205, 0}, {208, 207, 0}, {211, 208, 0},
					   {214, 208, 0}, {217, 208, 0}, {220, 208, 0},
					   {222, 207, 0}, {225, 205, 0}, {227, 204, 0},
					   {229, 202, 0}, {231, 199, 0}, {232, 197, 0},
					   {233, 194, 0}, {234, 191, 0}, {234, 188, 0},
					   {233, 185, 0}, {232, 183, 0}, {231, 180, 0},
					   {229, 178, 0}, {227, 176, 0}, {225, 174, 0},
					   {223, 172, 0}, {221, 170, 0}, {219, 168, 0},
					   {216, 166, 0}, {214, 164, 0}, {212, 162, 0},
					   {210, 161, 0}, {207, 159, 0}, {205, 157, 0},
					   {203, 156, 0}, {200, 154, 0}, {198, 152, 0},
					   {195, 151, 0}, {193, 149, 0}, {190, 148, 0},
					   {188, 147, 0}, {185, 145, 0}, {183, 144, 0},
					   {180, 143, 0}, {177, 141, 0}, {175, 140, 0},
					   {172, 139, 0}, {169, 138, 0}, {167, 137, 0},
					   {164, 136, 0}, {161, 135, 0}, {158, 134, 0},
					   {156, 133, 0}, {153, 132, 0}, {150, 132, 0},
					   {147, 131, 0}, {145, 130, 0}, {142, 130, 0},
					   {139, 129, 0}, {136, 128, 0}, {133, 128, 0},
					   {130, 127, 0}, {127, 127, 0}, {125, 127, 0},
					   {122, 127, 0}, {119, 127, 0}, {116, 127, 0},
					   {113, 127, 0}, {110, 128, 0}, {107, 128, 0},
					   {104, 128, 0}, {102, 127, 0}, { 99, 126, 0},
					   { 97, 124, 0}, { 95, 122, 0}, { 93, 120, 0},
					   { 92, 117, 0}, { 92, 114, 0}, { 92, 111, 0},
					   { 93, 108, 0}, { 94, 106, 0}, { 96, 104, 0},
					   { 98, 102, 0}, {100, 100, 0}, {103,	99, 0},
					   {106,  99, 0}, {109,  99, 0}, {111, 100, 0},
					   {114, 101, 0}, {117, 102, 0}, {120, 103, 0},
					   {123, 102, 0}, {125, 102, 0}, {128, 100, 0},
					   {130,  98, 0}, {132,  96, 0}, {133,	94, 0},
					   {134,  91, 0}, {134,  88, 0}, {134,	85, 0},
					   {133,  82, 0}, {131,  80, 0}, {129,	78, 0}
					};

					if (r >= 128) {
						int ix = (int) (((r - 128) * (ELEMENTS(pgnd) - 1)) / 127);

						/* Land area.  Look up colour based on elevation from
						   precomputed colour map table. */

						ir = pgnd[ix][0];
						ig = pgnd[ix][1];
						ib = pgnd[ix][2];
					} else {

						/* Water.  Generate clouds above water based on
						   elevation.  */

						ir = ig = (int) (r > 64 ? (r - 64) * 4 : 0);
						ib = 255;
					}

					/* Generate polar ice caps. */

					ice = max(0.0, (icet + glaciers * max(-0.5, (r - 128) / 128.0)));
					if	(ice > 0.125) {
						ir = ig = ib = 255;
					}

					/* Apply limb darkening by cosine rule. */

					{	double dx = 2 * (((planetSize / 2) - j) /
									((double) planetSize)),
							   dxsq = dx * dx,
							   ds, di, inx;
						double dsq, dsat;
						di = svx * dx + svy + svz * sqrt(1.0 - dxsq);
#define 				PlanetAmbient  0.05
						if (di < 0)
							di = 0;
						di = min(1.0, di);

						ds = sqrt(dxsq + dysq);
						ds = min(1.0, ds);

						/* Calculate  atmospheric absorption  based on the
						   thickness of atmosphere traversed by  light	on
						   its way to the surface. */

#define 				AtSatFac 1.0
						dsq = ds * ds;
						dsat = AtSatFac * ((sqrt(Atthick * Atthick - dsq) -
								sqrt(1.0 * 1.0 - dsq)) / athfac);
#define 				AtSat(x, y) x = (int) (((x) * (1.0 - dsat)) + (y) * dsat)
						AtSat(ir, 127);
						AtSat(ig, 127);
						AtSat(ib, 255);

						inx = PlanetAmbient + (1.0 - PlanetAmbient) * di;
						ir = (int) (ir * inx);
						ig = (int) (ig * inx);
						ib = (int) (ib * inx);
					}

					SET_PIXEL(i, planetX + j, ir, ig, ib);
				}

				/* Left stars offset */

#define StarClose		2
				for (j = ((planetSize / 2) - lcos) - StarClose; j < (planetSize / 2) - lcos; j++) {
					SET_PIXEL(i, planetX + j, 0, 0, 0);
				}

				/* Right star offset */

				for (j = ((planetSize / 2) + lcos) + 1;
					 j < ((planetSize / 2) + lcos) + StarClose; j++) {
					SET_PIXEL(i, planetX + j, 0, 0, 0);
				}
			}
		}
	}

tickerbail:;
	if (!stars) {
		if (cp != NULL) {
			free(cp);
		}
		if (u != NULL) {
			free(u);
			free(u1);
			free(bxf);
			free(bxc);
		}
	}
#undef TICKEROO_INTERVAL
}

/*	PLANET	--	Make a planet.	*/

static Boolean planet(void)
{
	float *a = NULL;
	int i, j;
	double rmin = 1e50, rmax = -1e50, rmean, rrange;

	if (!clouds) {
		//	Initialise black body RGB intensity table
		if (black_to_pixel == NULL) {
			double t, br, bb, bg;
			unsigned char *bp;

			bp = black_to_pixel = (unsigned char *) malloc((((BBODY_TEMP_MAX - BBODY_TEMP_MIN) /
									BBODY_TEMP_RESOLUTION) + 1) * 3);
			for (t = BBODY_TEMP_MIN; t <= BBODY_TEMP_MAX; t += BBODY_TEMP_RESOLUTION) {
				black_body_rgb(t, &br, &bg, &bb);
				*bp++ = (int) (br * 255);
				*bp++ = (int) (bg * 255);
				*bp++ = (int) (bb * 255);
			}
		}
	}

	if (!stars) {
		spectralsynth(&a, meshsize, 3.0 - fracdim);
		if (bailout()) {
			if (a != NULL) {
				free(a);
				a = NULL;
			}
		}
		if (a == NULL) {
			return FALSE;
		}

		/* Apply power law scaling if non-unity scale is requested. */

		if (powscale != 1.0) {
			for (i = 0; i < meshsize; i++) {
				for (j = 0; j < meshsize; j++) {
				   double r = Real(a, i, j);

					if (r > 0) {
						Real(a, i, j) = (float) pow(r, powscale);
					}
				}
			}
		}

		/* Compute extrema for autoscaling. */

		for (i = 0; i < meshsize; i++) {
			for (j = 0; j < meshsize; j++) {
				double r = Real(a, i, j);

				rmin = min(rmin, r);
				rmax = max(rmax, r);
			}
		}
		rmean = (rmin + rmax) / 2;
		rrange = (rmax - rmin) / 2;
		for (i = 0; i < meshsize; i++) {
			for (j = 0; j < meshsize; j++) {
				Real(a, i, j) = (float) ((Real(a, i, j) - rmean) / rrange);
			}
		}
	}
	genplanet(a, meshsize);
	if (a != NULL) {
		free(a);
	}
	return TRUE;
}

//	FORGE_INIT  --  One-time initialisation

void forge_init(void)
{
	initgauss();
}

//	FORGE_TERM  --  One-time termination

void forge_term(void)
{
	if (black_to_pixel != NULL) {
		free(black_to_pixel);
		black_to_pixel = NULL;
	}
}

/*	SYNTH_SETUP  --  Set up parameters for synthesis of the next image.  */

void synth_setup(void)
{
	int i, sy, yo;

	screenxsize = screenX;
	screenysize = sy = screenY;

	/* Force  screen to be at least  as wide as it is high.  Long,
	   skinny screens  cause  crashes  because	picture  width	is
	   calculated based on height.	*/

	screenxsize = max(screenysize, screenxsize);
	screenxsize = (screenxsize + 1) & (~1);

	yo = 0;
	if (showDateTime && !fChildPreview) {
		sy -= tmx.tmHeight + 1;
		yo += tmx.tmHeight + 1;
	}
	planetSize = (min(screenxsize, sy) * 3) / 4;

	if (randomPosition) {
		planetX = random() % (screenxsize - planetSize);
		planetY = yo + (random() % (sy - planetSize));
	} else {
		planetX = (screenxsize - planetSize) / 2;
		planetY = (screenysize - planetSize) / 2;
	}
	lastPlanetSize = planetSize;
	lastPlanetX = planetX;

	/* Set defaults when explicit specifications were not given.

	   The	default  fractal  dimension  and  power  scale depend upon
       whether we're generating a planet or clouds. */

	arand = pow(2.0, 15.0) - 1.0;

	clouds = stars = hell = FALSE;

	if (show_planets) {
		if (show_clouds && ((random() & 7) == 3)) {
			clouds = TRUE;
		} else if (show_stars && ((random() & 7) == 5)) {
			stars = TRUE;
		}
	} else {
		clouds = (!show_stars) || (show_clouds && (random() & 1));
		stars = !clouds;
	}

	//	Set parameters for this generation

#define CastParameter(param)	Cast(param##_min, param##_max)

	fracdim = clouds ? CastParameter(fractal_dimension_clouds) : \
					   CastParameter(fractal_dimension);
	powscale = clouds ? CastParameter(power_spectrum) : \
						CastParameter(power_spectrum_clouds);
	icelevel = CastParameter(ice_level);
	glaciers = CastParameter(glacier_level);
	starfraction = (int) CastParameter(star_fraction);
	starcolour = (int) CastParameter(star_colour);

	nclusters = (int) Cast(star_clusters_min, star_clusters_max + 1);
	/*	We allow setting nclusters negative so the user can
		select clusters in only a given percentage of images. */
	if (nclusters < 0) {
		nclusters = 0;
	} else if (nclusters > MAX_STAR_CLUSTERS) {
		/*	This shouldn't be possible, but better safe than subscripting
			off the end of the clusters[] array.  */
		nclusters = MAX_STAR_CLUSTERS;
	}
	for (i = 0; i < nclusters; i++) {
		clusters[i].clusterpos[X] = (int) Cast(0, screenxsize);
		clusters[i].clusterpos[Y] = (int) Cast(0, screenysize);
		clusters[i].clusterdens = (int) CastParameter(star_cluster_density);
		clusters[i].clusterwid = (int) (min(screenxsize, screenysize) *
										(CastParameter(star_cluster_width) / 100.0));
	}

	meshsize = mesh_size;
}

//	SYNTHESISE_IMAGE  --  Create synthetic image with current settings

int synthesise_image(HGLOBAL *LimageBitmap, int *LsizeX, int *LsizeY)
{
	LPBITMAPINFOHEADER bh;
	DWORD bmpsize;
	LPBYTE pix;
	int linewid;
	int pixbytes;
	static LPBYTE sl = NULL;
	static HGLOBAL imageBitmap = NULL;		// In-memory bitmap

	sl = NULL;
	imageBitmap = NULL;

	pixbytes = 3;

	linewid = ((((screenxsize * pixbytes) + (sizeof(LONG) - 1)) / sizeof(LONG)) * sizeof(LONG));
	bmpsize = sizeof(BITMAPINFOHEADER) + (linewid * screenysize);

	imageBitmap = malloc(bmpsize);
	if (imageBitmap == NULL) {
		*LimageBitmap = NULL;
		loadError = "Cannot allocate bitmap for image";
		free(sl);
		return FALSE;
	}

	//	Plug in bitmap header fields

	bh = (LPBITMAPINFOHEADER) imageBitmap;
	pix = ((LPBYTE) imageBitmap) + sizeof(BITMAPINFOHEADER);
	bh->biSize = sizeof(BITMAPINFOHEADER);
	bh->biWidth = screenxsize;
	bh->biHeight = screenysize;
	bh->biPlanes = 1;
	bh->biBitCount = 24;
	bh->biCompression = BI_RGB;
	bh->biSizeImage = 0;
	bh->biXPelsPerMeter = bh->biYPelsPerMeter = 2835;
	bh->biClrUsed = 0;
	bh->biClrImportant = 0;

	pixelArray = pix;
	pixelLineLength = linewid;

	planet();								// Generate image
	if ((loadError == NULL) && bailout()) {
		loadError = "Bailed out due to mouse or keyboard event";
	}

	if (loadError != NULL) {
		free(imageBitmap);
		*LimageBitmap = NULL;
		return FALSE;
	}

#ifdef SNAP_BMP
{
	FILE *o_file = fopen("c:/tmp/snap.bmp", "wb");

	BITMAPFILEHEADER bf;

	memcpy(&bf.bfType , "BM", 2);
	bf.bfSize = (sizeof bf) + bmpsize;
	bf.bfReserved1 = bf.bfReserved2 = 0;
	bf.bfOffBits = (sizeof bf) + sizeof(BITMAPINFOHEADER) + (usePalette * 256 * sizeof(RGBQUAD));
	fwrite(&bf, sizeof bf, 1, o_file);
	fwrite(bh, bmpsize, 1, o_file);
	fclose(o_file);
}
#endif
	 
	*LsizeX = (int) bh->biWidth;
	*LsizeY = (int) bh->biHeight;
	*LimageBitmap = imageBitmap;
	return TRUE;
}

