#! /usr/bin/perl

#                            L o g J a m
#
#       E-mail warnings when log files exceed specified sizes
#
#                           by John Walker
#                      http://www.fourmilab.ch/
#
#               This program is in the public domain.

my $Version =           '1.1 -- October 2017';

    use strict;
    use warnings;

#   Site-wide configuration

    my $Notify = 'root@localhost';    # Address(es) to which warnings should be E-mailed
    my $Who = 'carbon-based lifeform';# Name to which messages should be addressed

#   User options

    my $nomail = 0;                   # Send output to STDOUT instead of mailing to administrator
    my $thousands_size = 0;           # Edit size with thousands separator ?

#   Localisation options

    my $Thousands = ',';              # Thousands separator or '' for none

#   Configuration for the particulars of the system on which we're running

    my $du_command = 'du -sk';        # Command to obtain size of directory contents
    my $expat = '(\d+)\D';            # Pattern to extract percentage used
    my $du_blocksize = 1024;          # Block size returned by $du_command
    my $mail = 'Mail -s';             # Command to mail with subject on command line
                                      # On some systems you may have to change this
                                      # to 'mailx -s'.
    my $hostname = `hostname`;        # Identity of machine
    chop($hostname);

#   You shouldn't need to change anything after this line.

    my ($opt_d, $opt_g, $opt_m, $opt_t, $opt_u);
    Getopts('dg:m:tu');

    if (defined $opt_d) {             # -d  Debug mode: output to STDOUT
        $nomail = 1;
    }

    if (defined $opt_g) {             # -g  Greeting name
        $Who = $opt_g;
    }

    if (defined $opt_m) {             # -m address  Set E-mail address for report
        $Notify = $opt_m;
    }

    if (defined $opt_t) {             # -t  Edit sizes with thousands separator
        $thousands_size = 1;
    }

    if (defined $opt_u) {             # -u  Print how to call information
        print STDERR << "EOF";
Usage: LogJam.pl [ options ] filename threshold ...
    Options:
        -d                   Debug mode: output to STDOUT instead of mail
        -g name              Name to whom greeting should be addressed
        -m address           E-mail report to address
        -t                   Edit sizes with thousands separator
        -u                   Print this message
Version $Version
EOF
        exit(0);
    }

    my $n = 0;
    my @bloviation;

    #   Iterate over the remaining arguments, which are names of
    #   files or directories and the warning threshold for each.

    for (my $i = 0; $i <= $#ARGV; $i += 2) {
        my $filename = $ARGV[$i];
        my $threshold = ScanSize($ARGV[$i + 1]);

        my $tred = comma($threshold);
        if (-e $filename) {
            if (-f $filename) {
                my $filelength = -s $filename;
                if ($filelength >= $threshold) {
                    my $fled = comma($filelength);
                    $bloviation[$n++] = "$filename ($fled bytes) exceeds threshold of $tred bytes.";
                }
            } elsif (-d $filename) {
                my $dsize = `$du_command $filename`;
                $dsize =~ m@$expat@;
                my $filelength = $1 * $du_blocksize;
                if ($filelength >= $threshold) {
                    my $fled = comma($filelength);
                    $bloviation[$n++] = "$filename contents ($fled bytes) exceeds threshold of $tred bytes.";
                }
            } else {
                $bloviation[$n++] = "$filename is a neither a plain file nor directory.";
            }
        } else {
            $bloviation[$n++] = "$filename does not exist.";
        }
    }

    #   If any file systems exceed their threshold, post a heads-up
    #   to the administrator.

    if ($n > 0) {
        my $hostmsg = "'File size warning for $hostname'";
        my $mailto = "$mail $hostmsg $Notify";
        if ($nomail) {
            open(OF, ">-") || die("Cannot open standard output");
            print(OF "$mailto\n");
        } else {
            open(OF, "| $mailto") || die("Cannot open pipe to $mailto");
        }

        print(OF "Greetings, $Who.  The following files on\n");
        print(OF "$hostname exceed their size threshold or have errors.\n\n");
        foreach my $message (@bloviation) {
            print(OF "$message\n");
        }
        close(OF);
    }

#   Scan a size quantity

sub ScanSize {
    my ($sizespec) = @_;
    my ($num, $suffix, $unit);

    $unit = 1;
    $sizespec =~ m/(\d+)(\w)/;
    $num = $1;
    $suffix = $2;
    $suffix =~ tr/A-Z/a-z/;

    if ($suffix eq 'b') {
        $unit = 512;
    } elsif ($suffix eq 'k') {
        $unit = 1000;
    } elsif ($suffix eq 'm') {
        $unit = 1000000;
    } elsif ($suffix eq 'g') {
        $unit = 1000000000;
    } elsif ($suffix eq 't') {
        $unit = 1000000000000;
    }

    return ($num * $unit);
}

sub comma {
    my ($v, $o);

    $v = sprintf("%.0f", $_[0]);
    $o = "";
    if ($Thousands ne '') {
        while (length($v) >= 4) {
            $o = $Thousands . substr($v, -3, 3) . $o;
            $v = substr($v, 0, length($v) - 3);
        }
    }
    return ($v . $o);
}

# getopts.pl - a better getopt.pl

# Usage:
#      Getopts('a:bc');  # -a takes arg. -b & -c not. Sets opt_* as a
#                        #  side effect.

#   This is equivalent to the Perl core module Getopt::Std's
#   getopts() function.  It was embedded to avoid path lookup
#   problems when Perl is run from cron jobs or by root.

sub Getopts {
    my ($argumentative) = @_;
    my (@args, $first, $rest, $pos);
    my ($errs) = 0;

    @args = split( / */, $argumentative );
    while (@ARGV && ($_ = $ARGV[0]) =~ /^-(.)(.*)/) {
        ($first, $rest) = ($1, $2);
        $pos = index($argumentative, $first);
        if ($pos >= 0) {
            if (($args[$pos + 1]) && ($args[$pos + 1] eq ':')) {
                shift(@ARGV);
                if ($rest eq '') {
                    ++$errs unless @ARGV;
                    $rest = shift(@ARGV);
                }
                eval "\$opt_$first = \$rest;";
            } else {
                eval "\$opt_$first = 1";
                if ($rest eq '') {
                    shift(@ARGV);
                } else {
                    $ARGV[0] = "-$rest";
                }
            }
        } else {
            print(STDERR "Unknown option: $first\n");
            ++$errs;
            if ($rest ne '') {
                $ARGV[0] = "-$rest";
            } else {
                shift(@ARGV);
            }
        }
    }
    return ($errs == 0);
}
