#! /usr/bin/perl

#                         W a t c h F u l l
#
#        E-mail warnings when file systems approach capacity
#
#                           by John Walker
#                      http://www.fourmilab.ch/
#
#               This program is in the public domain.

my $Version =           '1.1 -- October 2017';

    use strict;
    use warnings;

#   Site-wide configuration

    my $Notify = 'root@localhost';    # Address(es) to which warnings should be E-mailed
    my $Who = 'carbon-based lifeform';# Name to which messages should be addressed

#   Configuration for specific file system properties on this machine

    my $Threshold = 90;               # Default percent-full warning threshold

#   User options

    my $nomail = 0;                   # Send output to STDOUT instead of mailing to administrator

#   Configuration for the particulars of the system on which we're running

    my $df_command = 'df -k';         # Command to obtain file system status
    my $expat = qr'(\d+)\D*\s\s*(/.*)$'; # Pattern to extract percentage used
    my $mail = 'Mail -s';             # Command to mail with subject on command line
                                      # On some systems you may have to change this
                                      # to 'mailx -s'.
    my $hostname = `hostname`;        # Identity of machine
    chop($hostname);

#   You shouldn't need to change anything after this line.

    my ($opt_d, $opt_g, $opt_m, $opt_s, $opt_t, $opt_u, $opt_x);
    Getopts('dg:m:s:t:ux:');

    my (%Exclude, %FSThreshold, @bulging);

    if (defined $opt_d) {             # -d  Debug mode: output to STDOUT
        $nomail = 1;
    }

    if (defined $opt_g) {             # -g  Greeting name
        $Who = $opt_g;
    }

    if (defined $opt_m) {             # -m address  Set E-mail address for report
        $Notify = $opt_m;
    }

    if (defined $opt_s) {             # -s fs=thresh,fs=thresh,...  Set threshold by file system
        my $xh;
        foreach $xh (split(/,/, $opt_s)) {
            if ($xh =~ m/(^[^=]+)=(\d+)$/) {
                $FSThreshold{$1} = $2;
            } else {
                print(STDERR "Syntax error in -s option string: $xh\n");
                exit(2);
            }
        }
    }

    if (defined $opt_t) {             # -t thresh  Set default threshold
        $Threshold = $opt_t;
    }

    if (defined $opt_u) {             # -u  Print how to call information
        print STDERR << "EOF";
Usage: WatchFull.pl [ options ]
    Options:
        -d                   Debug mode: output to STDOUT instead of mail
        -g name              Name to whom greeting should be addressed
        -m address           E-mail report to address
        -s /path=thresh,...  Set warning thresholds for specific file systems
        -t thresh            Set default warning threshold
        -u                   Print this message
        -x /path,...         Exclude specified mount points from check
Version $Version
EOF
        exit(0);
    }

    if (defined $opt_x) {             # -x fs,fs,...  Exclude file systems
        my $xh;
        foreach $xh (split(/,/, $opt_x)) {
            $Exclude{$xh} = 1;
        }
    }

    open(DF, "$df_command |") || die("Cannot open pipe to $df_command");

    my $n = 0;
    my $f;

fs: while ($f = <DF>) {
        my ($frac, $fs, $thresh);

        chop($f);

        if ($f =~ m@$expat@) {
            $frac = $1;
            $fs = $2;

            #   See if the file system is explicitly excluded

           if (%Exclude && $Exclude{$fs}) {
                next fs;
            }

            #   Determine the threshold for this file system, either
            #   explicitly specified or default.

            $thresh = $Threshold;
            if ($FSThreshold{$fs}) {
                $thresh = $FSThreshold{$fs};
            }

            #   Test whether this included file system is, in fact,
            #   over its warning threshold.  If so, save the
            #   particulars for mailing to the system administrator.

            if ($frac >= $thresh) {
                $bulging[$n++] = $f;
            }
        }
    }
    close(DF);

    #   If any file systems fall within the threshold, post a heads-up
    #   to the administrator.

    if ($n > 0) {
        my $hostmsg = "'File system capacity warning for $hostname'";
        my $mailto = "$mail $hostmsg $Notify";
        if ($nomail) {
            open(OF, ">-") || die "Cannot open standard output";
            print(OF "$mailto\n");
        } else {
            open(OF, "| $mailto") || die "Cannot open pipe to $mailto";
        }

        print(OF "Greetings, $Who.  The following file systems on\n");
        print(OF "$hostname are approaching capacity.\n\n");
        for (my $i = 0; $i <= $#bulging; $i++) {
            print(OF "$bulging[$i]\n");
        }
        close(OF);
    }

# getopts.pl - a better getopt.pl

# Usage:
#      Getopts('a:bc');  # -a takes arg. -b & -c not. Sets opt_* as a
#                        #  side effect.

#   This is equivalent to the Perl core module Getopt::Std's
#   getopts() function.  It was embedded to avoid path lookup
#   problems when Perl is run from cron jobs or by root.

sub Getopts {
    my ($argumentative) = @_;
    my (@args, $first, $rest, $pos);
    my ($errs) = 0;

    @args = split( / */, $argumentative );
    while (@ARGV && ($_ = $ARGV[0]) =~ /^-(.)(.*)/) {
        ($first, $rest) = ($1, $2);
        $pos = index($argumentative, $first);
        if ($pos >= 0) {
            if (($args[$pos + 1]) && ($args[$pos + 1] eq ':')) {
                shift(@ARGV);
                if ($rest eq '') {
                    ++$errs unless @ARGV;
                    $rest = shift(@ARGV);
                }
                eval "\$opt_$first = \$rest;";
            } else {
                eval "\$opt_$first = 1";
                if ($rest eq '') {
                    shift(@ARGV);
                } else {
                    $ARGV[0] = "-$rest";
                }
            }
        } else {
            print(STDERR "Unknown option: $first\n");
            ++$errs;
            if ($rest ne '') {
                $ARGV[0] = "-$rest";
            } else {
                shift(@ARGV);
            }
        }
    }
    return ($errs == 0);
}
