#! /usr/bin/perl

    #                 Make Decisions

    #       John Walker            October 2012

    #       Revised and updated    July 2017

=head1 NAME

decide, decidebin, dice, eightball - A just machine, to make big decisions

=head1 SYNOPSIS

B<decide|decidebin|dice|eightball>
[B<--apikey> I<key>]
[B<--binary>]
[B<--decide>]
[B<--dice> [I<expr>]]
[B<--eightball|8>]
[B<--face>]
[B<--help>]
[B<--hotbits>]
[B<--quiet>]
[B<--rand>]
[B<--urandom>]
[B<--verbose>]
I<dice_expr>

=head1 DESCRIPTION

B<decide>, B<decidebin>, B<dice>, and B<eightball>, all aliases
of the Perl program B<decide.pl>, make decisions for users,
using random or pseudorandom numbers from a variety of sources.

B<decide> and B<decidebin> make yes or no decisions.  B<decide>
responds "Yes" or "No" on standard output, while B<decidebin>
answers with "1" or "0".

B<eightball> responds with one of the twenty answers with
which the L<Magic 8-Ball|https://en.wikipedia.org/wiki/Magic_8-Ball>
toy replies to queries.

B<dice> rolls dice using the notation from role-playing
and war games and shows the result of the roll.

Random data can be obtained from the Fourmilab
L<HotBits|http://www.fourmilab.ch/hotbits/> radioactive
random number generator in either pseudorandom or genuine
random mode, from the system's B</dev/urandom> generator (if
available), or from Perl's built-in C<rand()> function.
Options allow selecting which generator is used.

=head1 OPTIONS

All options may be abbreviated to their shortest
unambiguous prefix.  Single-letter mode abbreviations
may be aggregated after a single dash.

=over 5

=item B<--apikey> I<key>

When querying the HotBits random number generator, the specified
I<key> is used as the API Key.  The HotBits generator requires
an API Key when requesting radioactively-generated random data.
If no API Key is specified or I<key> is C<Pseudorandom>, HotBits
will return pseudorandom data generated by a high quality
algorithm seeded with radioactively-generated data.

=item B<--binary>

Binary mode output ("1" or "0") is printed on standard output.

=item B<--decide>

"Yes" or "No" is printed on standard output.

=item B<--dice> I<expr>

Roll dice to evaluate I<expr> in dice notation, print
rolls and result on standard output.

=item B<--eightball> or B<--8>

One of the twenty Magic 8-Ball responses is printed on
standard output.

=item B<--face>

Show rolls of six-sided dice as Unicode die faces.

=item B<--help>

Display how to call information.

=item B<--hotbits>

The HotBits generator is selected.  If you require
radioactively-generated random data, you must specify a
valid HotBits API Key with the B<--apikey> option, otherwise
pseudorandom data will be used.

=item B<--quiet>

No output is printed.  The exit status will be set to reflect
the result: 1 or 0 for B<--decide> or B<--decidebin>, 0 to 19
for B<--eightball>, and the result of evaluating the dice
expression for B<--dice>.

=item B<--rand>

Perl's built-in C<rand()> generator is used.  The quality
of this generator depends upon the C language implementation
upon which Perl is built; it may be quite poor.

=item B<--urandom>

The system's B</dev/urandom> generator will be used, if
available.  This is usually a high-quality pseudorandom
generator seeded with hardware-generated entropy.

=item B<--verbose>

Debugging information is printed on standard error.

=back

=head1 GENERATOR SELECTION

If none of the B<--hotbits>, B<--urandom>, or B<--rand>
options are specified, those generators will be tried,
in that order, and the first to successfully return a
value will be used.  The B<--rand> generator will always
succeed, and is used if other options fail.  The
B<--verbose> option will show you which generator was
used and what it returned.

=head1 DICE EXPRESSIONS

When called as B<dice> or with the B<--dice> option, an
expression in
L<dice notation|https://en.wikipedia.org/wiki/Dice_notation>
is evaluated, obtaining the result of the dice rolls from
the selected generator.  Dice notation, commonly used in
role-playing and war games, expresses rolls of dice of
various kinds and ways of combining them.  Rolling
B<A> B<X>-sided dice is written as B<AdX> (the "B<d>"
may be upper or lower case).  For example, the roll of
two six-sided dice is written as "B<2d6>", and the result
is the sum of the top faces of the two dice, 2 to 12.  If
B<A> is omitted, 1 is assumed, and if B<X> is omitted, 6
is assumed.  When multiple dice are rolled, two lines of
output are printed: the first shows the results for the
individual dice (identified by the number of faces if
different kinds of dice are involved in the roll),
while the second shows their sum.  For example:

    $ dice 3d6+d8
    d6[ 2 3 6 ] d8[ 2 ]
    13

Dice can have any number of faces from 2 (flipping a coin)
to 256, with faces numbered from 1 to the number of faces.
All dice are fair: all faces have an equal probability of
coming up.  You can abbreviate a 100-faced die, often used
to compute percentages in games, as "B<d%>".

Multiple sets of rolls can be combined with arithmetic
operators into expressions.  For example, B<3d8-2> means
roll three 8-sided dice, sum their values, then subtract
two, and B<2d6+1d12> returns the sum of rolling two 6-sided
and one 12-sided dice.  Values can be combined with any
of the standard arithmetic operators: B<+>, B<->, B<*>,
and B</>.  You can write multiplication as B<x> or the
ISO/Unicode times character, and division as the ISO/Unicode
divide character.  Parentheses may be used to group complex
expressions, for example B<((2d6-2)+d4)*3>.  Expressions
which use characters to which the shell is sensitive, such
as parentheses and B<*>, must be quoted on the command line.

Rolls can select from the highest or lowest values in a
set of dice.  Specifying B<5d6h3> rolls 5 six-sided dice
and returns the sum of the three highest values among the five.
Similarly, B<5d6l3> sums the three lowest values.  If the
number is omitted after the B<h> or B<l>, 1 is assumed:
B<3dh> rolls three six-sided dice (the 6 is assumed as
no number if given after the B<d>) and chooses the highest
value rolled.  You can also specify selection in terms of
dropping values.  (Why?  Because gamers think in odd ways.)
A specification of B<4d8-l2> means roll four 8-sided dice,
drop the lowest two values, and sum the remaining two;
use B<-h> to drop the highest values.  Again, 1 is assumed
if no number to drop is given.

=head1 FILES

Output is written on standard output unless the
B<--quiet> option is specified.  Debugging output from
the B<--verbose> option is written on standard error.

=head1 BUGS

Unless the B<--verbose> option is set, there is no
indication if the HotBits or B</dev/urandom>
generators fail and the program falls back to Perl's
C<rand()> generator.

In order to use the HotBits random source, the
B<LWP::Simple> module must be installed.

In order to use the B<--urandom> source, the operating
system must implement the B</dev/urandom> pseudo
device file.  Non-Unix-like systems may not provide this
facility.

Please report bugs to B<bugs@fourmilab.ch>.

=head1 AUTHOR

John Walker
(B<http://www.fourmilab.ch/>)

=head1 SEE ALSO

    LWP::Simple
    /dev/urandom
    rand()
    http://www.fourmilab.ch/hotbits/
    http://www.perl.org/

=head1 VERSION

This is B<decide> version 1.0, released on July 6th, 2017.
The current version of this program is always posted at:

http://www.fourmilab.ch/webtools/decide/

=head1 COPYRIGHT

This program is in the public domain.

=cut

    use strict;
    use warnings;
    use Getopt::Long qw(GetOptions);
    Getopt::Long::Configure ("bundling");

    #   Program version and date
    my $version = '1.0 (July 6th, 2017)';

    #   To use radioactively-generated random numbers from
    #   HotBits:  http://www.fourmilab.ch/hotbits/
    #   specify the HotBits API key below.  If you specify
    #   an API key of 'Pseudorandom', HotBits-generated
    #   pseudorandom data (seeded with radioactively-generated
    #   data) will be used.  If $APIkey is the null string,
    #   random data from /dev/urandom (if available) or from
    #   Perl's built-in rand() function will be used instead.

    my $APIkey = 'Pseudorandom';

    #   URL to request HotBits
    my $Hotbits = 'https://www.fourmilab.ch/cgi-bin/Hotbits';

    #   Kernel random source
    my $randev = "/dev/urandom";

    #   Decide replies
    my @decide = ( 'No', 'Yes' );

    #   Binary replies
    my @binary = ( '0', '1' );

    #   Magic eightball replies
    my @eightball = (
        'It is certain',            #  0
        'It is decidedly so',       #  1
        'Without a doubt',          #  2
        "Yes \x{2014} definitely",  #  3
        'You may rely on it',       #  4
        'As I see it, yes',         #  5
        'Most likely',              #  6
        'Outlook good',             #  7
        'Yes',                      #  8
        'Signs point to yes',       #  9

        'Reply hazy, try again',    # 10
        'Ask again later',          # 11
        'Better not tell you now',  # 12
        'Cannot predict now',       # 13
        'Concentrate and ask again',# 14

        'Don\'t count on it',       # 15
        'My reply is no',           # 16
        'My sources say no',        # 17
        'Outlook not so good',      # 18
        'Very doubtful'             # 19
                    );

    my $callname = $0;
    if ($callname =~ m/(\w+)(?:\.\w*)?$/) {
        $callname = $1;
    } else {
        $callname = "";
    }

    my @replies;
    my ($verbose, $quiet);
    my ($optbin, $optdec, $opt8) = (0) x 3;
    my ($hotbits, $urandom, $rand, $dice, $d6face) = (0) x 5;
    my $dicexp;

    GetOptions( "apikey|a=s"    => \$APIkey,
                "binary|b"      => sub { $optbin = 1; @replies = @binary; },
                "decide|d"      => sub { $optdec = 1; @replies = @decide; },
                "dice:s"        => \$dicexp,
                "eightball|e|8" => sub { $opt8 = 1; @replies = @eightball; },
                "face|f"        => \$d6face,
                "help|?"        => sub { help(); exit(0); },
                "hotbits|h"     => \$hotbits,
                "quiet|q"       => \$quiet,
                "rand|r"        => \$rand,
                "urandom|u"     => \$urandom,
                "verbose|v"     => \$verbose) ||
        die("Invalid command line");
        
    #   If we were called as "dice" and no "--dice" argument was
    #   given, but there's a regular argument, treat it as the
    #   dice expression to be evaluated.
    if ($callname eq 'dice') {
        if (!defined($dicexp) && (scalar(@ARGV) > 0)) {
            $dicexp = shift(@ARGV);
        }
    }

    #   If a "--dice" option was specified, giving a dice
    #   expression, select dice mode.
    if (defined($dicexp) || ($callname eq 'dice')) {
        $dice = 1;
    }
    
    if ($dice && (($optbin + $optdec + $opt8) > 0)) {
        die("Cannot combine --dice with other output options");
    }

    #   If dice mode has been selected, but no expression was
    #   given, cast two d6.
    if ($dice && (!$dicexp)) {
        $dicexp = '2d6';
    }
    
    #   If no random source was selected, try HotBits first, then
    #   /dev/urandom, and then fall back to rand().
    if (($hotbits + $urandom + $rand) == 0) {
        $hotbits = $urandom = 1;
    }

    if ($verbose) {
        print(STDERR "Called as \"$callname\".\n");
    }

    #   Choose the form of output from the call name if not set by option
    if (!@replies) {
        if ($callname eq 'decide') {
            @replies = @decide;
        } elsif ($callname eq 'decidebin') {
            @replies = @binary;
        } elsif ($callname eq 'eightball') {
            @replies = @eightball;
        } else {
            @replies = @decide;
        }
    }

    binmode(STDOUT, ":utf8");

    if ($dice) {
        evaluateDice($dicexp);
    } else {
        my $r = intRandom(scalar(@replies));

        if ($quiet) {
            if ($verbose) {
                print(STDERR "Setting exit status to $r.\n");
            }
            exit($r);
        }

        print($replies[$r], "\n");
    }

    #   Return a uniformly distributed integer random value between
    #   0 and argument - 1.  Note that to avoid bias since the values
    #   returned by the generator are in the range 0-255 while the
    #   range 0 - ($limit - 1) are not necessarily commensurate, we
    #   may have to make multiple requests of the generator to obtain
    #   a value which has no bias taken mod $limit.  This, in practice,
    #   rarely happens.

    sub intRandom {
        my ($limit) = @_;

        if ($hotbits) {
            use LWP::Simple;

            while (1) {
                my $xml = get("$Hotbits?nbytes=1&fmt=xml&apikey=$APIkey");
                if (!($xml =~ m/<random\-data version="\d+\.\d+">\n\s*(\w\w)/)) {
                    last;
                }
                my $hexbyte = $1;
                my $byte = hex($hexbyte);
                my $try = $byte % $limit;
                if ($verbose) {
                    printf(STDERR "HotBits: %02X  %s  %d\n", $byte, $hexbyte, $try);
                }
                return $try if (($byte - $try) + ($limit + 1)) >= 0;
            }

            if ($verbose) {
                print(STDERR "Unable to obtain data from HotBits.\n");
            }
        }

        if ($urandom) {
            if (open(RI, "<$randev")) {
                my $c;
                while (read(RI, $c, 1)) {
                    my $byte = ord($c);
                    my $try = $byte % $limit;
                    if ($verbose) {
                        printf(STDERR "$randev: %02X  %d\n", $byte, $try);
                    }
                    if ((($byte - $try) + ($limit + 1)) >= 0) {
                        close(RI);
                        return $try;
                    }
                }
                close(RI);
            }

            if ($verbose) {
                print(STDERR "Unable to obtain data from $randev.\n");
            }
        }

        #   If we get here we fall back on Perl's built-in
        #   rand() function which, in turn, relies on C's.
        my $try = int(rand($limit));
        if ($verbose) {
            printf(STDERR "rand(): %d\n", $try);
        }
        return $try;
    }

    #   Evaluate an expression in dice notation

    sub evaluateDice {
        my ($dexp) = @_;

        my @diceRolls;      # List of all rolls performed
        my @rollMark;       # Mark rolls by primitive
        my @rollX;          # Mark rolls by kind of die

        if ($verbose) {
            print(STDERR "Expression ($dexp)\n");
        }

        my $odec = $dexp;
        
        #   Substitute alternative operator characters in expressions
        $odec =~ s/[\x{d7}xX]/*/g;      # &times;/x  => *
        $odec =~ s:\x{f7}:/:g;          # &divide; => /
        
        #   Expand percent notation into d100
        $odec =~ s/([dD])%/${1}100/g;

        #   Evaluate all primitive dice rolls within the expression

        while ($odec =~ s/\s*(\d*)[dD](\d*)(?:([\-])?([HhLl])(\d+)?)?\s*//) {
            my ($odecL, $odecR, $A, $X, $HLS, $HL, $HLC) =
                ($`, $', $1, $2, $3, $4, $5);
            my $zhls = $HLS ? $HLS : "";
            my $zhl = $HL ? "$HL" : "";
            my $zhlc = $zhl ? ($HLC ? $HLC : "") : "";
            my $zspec = "${A}d$X$zhls$zhl$zhlc";

            if ($verbose) {
                print(STDERR "Term: $zspec  Before($odecL)  After($odecR)\n");
            }

            if (!$A) {
                $A = 1;
            }
            if (!defined($X) || ($X eq '')) {
                $X = 6;
            }
            if (!defined($HLS)) {
                $HLS = "";
            }
            if (!defined($HLC)) {
                $HLC = 1;
            }

            $zhls = $HLS ? $HLS : "";
            $zhlc = $zhl ? ($HLC ? $HLC : "") : "";
            if ($verbose) {
                print(STDERR "Expand: ${A}d$X$zhls$zhl$zhlc\n");
            }

            if ($A < 1) {
                die("Invalid roll count $A in $zspec");
            }
            if (($X < 2) || ($X > 256)) {
                die("Invalid die face count $X in $zspec");
            }
            if ($zhl && ($HLC >= $A)) {
                die("Rank count $HLC greater or equal to roll count $A in $zspec");
            }

            my $rollStart = scalar(@diceRolls);     # Index of first of rolls

            #   Perform the requested number of rolls for this item

            for (my $i = 0; $i < $A; $i++) {
                push(@diceRolls, intRandom($X) + 1);
                push(@rollMark, ($i == 0) ? 1 : 0);
                push(@rollX, $X);
            }
            $rollMark[$#rollMark] |= 2;

            #   If this is a summing specification, add them up

            my $result = 0;
            if (!$HL) {
                for (my $i = 0; $i < $A; $i++) {
                    $result += $diceRolls[$rollStart + $i];
                }
                if ($verbose) {
                    print(STDERR "(");
                    for (my $i = 0; $i < $A; $i++) {
                        print(STDERR " $diceRolls[$rollStart + $i] ");
                        if ($i < ($A - 1)) {
                            print(STDERR "+");
                        }
                    }
                    print(STDERR ") = $result\n");
                }
            } else {

                #   This is a greatest/least specification.  Start by
                #   sorting the rolls into numerical order.

                my @rolls = @diceRolls;         # All rolls so far
                splice(@rolls, 0, $rollStart);  # Remove rolls by previous expressions
                @rolls = sort({ $a <=> $b } @rolls);   # Sort numerically
                if ($verbose) {
                    print(STDERR "Sorted: [");
                    for (my $i = 0; $i < $A; $i++) {
                        print(STDERR " $rolls[$i] ");
                        if ($i < ($A - 1)) {
                            print(STDERR ",");
                        }
                    }
                    print(STDERR "]\n");
                }

                #   Transform the H/L specification into canonical
                #   form.

                my $sel = $HLC;                 # Number to select or drop
                my $top = ($HL =~ m/[hH]/);     # Top or bottom values ?
                if ($HLS eq '-') {              # If sign is negative...
                    $top = !$top;               # ...invert sense of selection
                    $sel = $A - $sel;           # ...subtract drop count to get selects.
                }
                if ($verbose) {
                    print(STDERR "Select the " . ($top ? "top" : "bottom") .
                        " $sel of $A\n");
                }
                if ($top) {                     # Ditch rolls which didn't make the cut...
                    splice(@rolls, 0, $A - $sel);    #      ... top wanted: drop lowest.
                } else {
                    splice(@rolls, $sel, $A - $sel); # ... bottom wanted: drop highest.
                }
                #   Sum the remaining values to yield the result
                map({ $result += $_ } @rolls);
                if ($verbose) {
                    print(STDERR "Selected: [");
                    for (my $i = 0; $i < scalar(@rolls); $i++) {
                        print(STDERR " $rolls[$i] ");
                        if ($i < (scalar(@rolls) - 1)) {
                            print(STDERR "+");
                        }
                    }
                    print(STDERR "] = $result\n");
                }
            }
            $odec = $odecL . $result . $odecR;

            if ($verbose) {
                print(STDERR "Roll result: $odec\n");
            }
        }
        
        #   Now we have evaluated all of the roll expressions in
        #   the argument.  Evaluate the result, which evaluates
        #   arithmetic expressions involving the rolls and
        #   constants.
        
        my $evex = int(eval($odec));
        if ($@) {
            die("Syntax error evaluating dice expression $dexp = $odec");
        }
        
        #   If --quiet, just set exit status to final expression value
        if ($quiet) {
            if ($verbose) {
                print(STDERR "Setting exit status to $evex.\n");
            }
            exit($evex);
        }
        
        #   If we've made more than one roll, show the individual
        #   rolls, grouped by primitive terms.
        
        if (scalar(@diceRolls) > 1) {
            #   See if we've rolled more than one kind of die.
            #   If so, we need to identify the rolls by type.
            my $multDie = 0;
            for (my $i = 1; $i < scalar(@diceRolls); $i++) {
                if ($rollX[$i] != $rollX[0]) {
                    $multDie++;
                    last;
                }
            }
            for (my $i = 0; $i < scalar(@diceRolls); $i++) {
                if ($i > 0) {
                    print(" ");
                }
                if ($rollMark[$i] & 1) {
                    if ($multDie) {
                        print("d$rollX[$i]");
                    }
                    print("[ ");
                }
                #   If face display is selected, show Unicode die face character
                if ($d6face && ($rollX[$i] == 6)) {
                    print(chr(9855 + $diceRolls[$i]) . "/$diceRolls[$i]");
                } else {
                    print("$diceRolls[$i]");
                }
                if ($rollMark[$i] & 2) {
                    print(" ]");
                }
            }
            print("\n");
        }
        
        print("$evex\n");
    }

    #   Print help text

    sub help {
        print("$callname [ options ]" .
                (($callname eq 'dice') ? " [ expr ]" : "") . "\n");
        print("Options:\n");
        print("    --apikey key     Specify HotBits API key\n");
        print("    --binary         Binary (1/0) output\n");
        print("    --decide         Decide (Yes/No) output\n");
        print("    --dice expr      Evaluate dice notation expr\n");
        print("    --eightball/8    Answer like Magic 8-Ball\n");
        print("    --face           Show die faces for d6 rolls\n");
        print("    --help           Print this message\n");
        print("    --hotbits        Use the HotBits generator\n");
        print("    --quiet          No text output, set exit status\n");
        print("    --rand           Use Perl rand() generator\n");
        print("    --urandom        Use /dev/urandom generator\n");
        print("    --verbose        Print debugging output\n");
        print("Version $version\n");
    };
