/*         Null document processor

           K. R. Throop  -  5 Feb 85

           Enhanced to support command line options    KRT 24 Feb 85
           Page numbering and auto-LaserJet added      KRT  2 May 85
           Page n of m option on -N added              KRT  4 May 85
           Double spacing and line counting added      KRT 13 May 85
           Running headings added                      KRT 14 May 85
           Laser reset & proportional mode added       KRT 15 May 85
           Justification (for text, not existence)     KRT 16 May 85
           Automatic expansion of TC-compressed files  KRT 19 May 85
           Page counting and change bars               KRT 25 May 85
           Minus bar for deleted lines on -C output    KRT 26 May 85
           Programmer-style line numbering (-SP)       KRT 27 May 85
           AutoBOOK output format                      KRT  9 Jul 85
           Remove tabs, handle formfeed                KRT  3 Aug 85
           Prudent man defaults                        KRT 16 Oct 85
           Fixed centring of page numbers              KRT  8 Dec 85
           "<<" include file capability                KRT  8 Mar 86
           Centring, right, and left justification     KRT 17 Mar 86
           Embedded documentation in programs          KRT 17 Mar 86
           Fixed page counting bug                     KRT 18 Mar 86
           Automatic text analysis and formatting      KRT 19 Mar 86
           Justification bug fix                       KRT 24 Jul 89

*/

#define REVDATE "24th July 1989"

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <limits.h>

#define TRUE  1
#define FALSE 0

#define MAXIN  32767         /* Maximum input line length */
#define LPP    56            /* Default lines per page */
#define DEFIND 5             /* Default indentation */
#define EDPIND 20            /* Indentation for EDP paper */
#define PNOIND 35            /* Page number indentation */
#define TOPMRG 3             /* Lines to skip at top of page */
#define JUSTCOL 70           /* Justification output columns */
#define JUSTPER 80           /* Justification line fill criterion */
#define SYSTABS 8            /* System default tab setting */

#define FORMFEED "\14"       /* Feed page printer command */

#define FFCMD   '\14'        /* Form feed command character */
#define INCLUDE "<<"         /* Sentinel for include file command */
#define SENT1   ">>"         /* Sentinel 1 for justification control */
#define SENT2   "<<"         /* Sentinel 2 for justification control */
#define EMBDOC1 "/*DOC"      /* Embedded document start sentinel */
#define EMBDOC2 "DOC*/"      /* Embedded document end sentinel */
#define JCOL1   ">!"         /* Justification column sentinel 1 */
#define JCOL2   '!'          /* Justification column sentinel 2 (NOTE CHAR!) */

#define EOS '\0'

static char cbfr1[MAXIN], cbfr2[MAXIN], inds[132],
            pnos[132], lino[10], linull[10], rhead[132];
static char *tbfr, *nbfr;
static int lpu = 0;
static int lcount = 0;
static int indent = DEFIND;
static int pagelen = LPP;
static int topmarg = TOPMRG;
static int numpag = FALSE;
static int numlin = FALSE;
static int numprog = FALSE;
static int dblspc = FALSE;
static int formfeed = TRUE;
static int runhead = FALSE;
static int laser = FALSE;
static int widemode = FALSE;
static int prudent = FALSE;
static int bookmode = FALSE;
static int prop = FALSE;
static int just = FALSE;
static int expert = 2;
static int embprog = FALSE;
static int embact = FALSE;
static int pcount = FALSE;
static int jlen = JUSTCOL;
static int jperc = JUSTPER;
static int pageno = 0, ofpages = 0;
static int pnoi = PNOIND;
static int eof = FALSE;
static int aline = 0;
static FILE *chgf = NULL;

struct isource {                   /* Input source descriptor */
        struct isource *isnext;
        FILE *isfp;
};

struct isource *istack = NULL;

struct wd {                        /* Word buffer */
        struct wd *wdn, *wdp;
        int wlen;
        char wdelim;
        char wdt[2];
};

static struct wd *whn = NULL, *whp;

static void chkeop(), pgtop(), pgbot(), detab(), wordify();
static int justify(), jline(), getl(), getins(), getexp(), analyse();
static char *isbar();

int main(argc, argv)
  int argc; char *argv[];
{
        int i;
        char *cp, opt;

        for (i = 0; i < argc; i++) {
           cp = argv[i];
           if (*cp == '-') {
              opt = *(++cp);
              if (islower(opt))
                 opt = toupper(opt);
              switch (opt) {

                 case 'B':
                    bookmode = TRUE;
                    indent = 0;
                    runhead = FALSE;
                    break;

                 case 'C':
                    if ((chgf = fopen(cp + 1, "r")) == NULL) {
                       fprintf(stderr, "Cannot open change file %s", cp+1);
                       return 1;
                    }
                    break;

                 case 'D':
                    dblspc = TRUE;
                    break;

                 case 'H':
                    runhead = TRUE;
                    break;

                 case 'I':
                    indent = atoi(cp+1);
                    break;

                 case 'J':
                    just = TRUE;
                    if (*(cp+1)) {
                       jlen = atoi(cp+1);
                       pnoi = jlen / 2;
                    }
                    break;

                 case 'L':
                    laser = TRUE;
                    opt = *(cp+1);
                    if (islower(opt))
                       opt = toupper(opt);
                    if (opt == 'P')
                       prop = TRUE;
                    break;

                 case 'N':
                    numpag = TRUE;
                    opt = *(cp+1);
                    if (islower(opt))
                       opt = toupper(opt);
                    if (opt == 'C') {
                       pcount = TRUE;
                       break;
                    }
                    if (*(cp+1))
                       ofpages = atoi(cp+1);
                    break;

                 case 'P':
                    pagelen = atoi(cp+1);
                    break;

                 case 'S':
                    numlin = TRUE;
                    opt = *(cp+1);
                    if (islower(opt))
                       opt = toupper(opt);
                    if (opt == 'P')
                       numprog = TRUE;
                    break;

                 case 'T':
                    topmarg = atoi(cp+1);
                    break;

                 case 'W':
                    widemode = TRUE;
                    break;

                 case 'X':
                    if (*(cp + 1))
                       expert = atoi(cp + 1);
                    else
                       expert = FALSE;
                    break;

                 case 'Z':
                    prudent = TRUE;
                    break;

                 case '?':
                 case 'U':
        fprintf(stderr,"\nNDOC --  The null document processor.  Call");
        fprintf(stderr,"\n         with NDOC [options] <input >output.");
        fprintf(stderr,"\n");
        fprintf(stderr,"\n         Options:   -B   Pagination for AutoBOOK");
        fprintf(stderr,"\n                    -Cf  Change bars with DIFF file f");
        fprintf(stderr,"\n                    -D   Double space");
        fprintf(stderr,"\n                    -H   1st line is running heading");
        fprintf(stderr,"\n                    -In  Indent n columns");
        fprintf(stderr,"\n                    -Jn  Justify to column n");
        fprintf(stderr,"\n                    -Lx  Set up for LaserJet");
        fprintf(stderr,"\n                          P = Proportional space");
        fprintf(stderr,"\n                    -Nn  Number pages [n = \"of\"]");
        fprintf(stderr,"\n                          C = Count pages only");
        fprintf(stderr,"\n                    -Pn  Page length n");
        fprintf(stderr,"\n                    -Sx  Number lines lawyer-like");
        fprintf(stderr,"\n                          P = Programmer-like");
        fprintf(stderr,"\n                    -Tn  Top margin n");
        fprintf(stderr,"\n                    -W   Print on wide (EDP) paper");
        fprintf(stderr,"\n                    -Xn  Expert system level n");
        fprintf(stderr,"\n                    -Z   Defaults of a prudent man");
        fprintf(stderr,"\n");
        fprintf(stderr,"\n          (P) Throoput Ltd.  %s", REVDATE);
        fprintf(stderr,"\n                All Rights Reversed");
        fprintf(stderr,"\n");
                    return 0;
              }
           }
        }

        if (laser) {
           printf("\33E");
           if (prop)
              printf("\33(s1P\33(s10V");
        }
        if (widemode) {
           indent = EDPIND;
        }
        if (prudent) {
           runhead = just = numpag = TRUE;
           indent = 6;
           if (!ofpages)
              pcount = TRUE;
        }
        if (numpag) {
           if (pagelen >= 2)
             pagelen -= 2;
           pnos[0] = EOS;
           if (ofpages)
              pnoi -= 5;
           for (i = 0; i < pnoi; i++)
              strcat(pnos, " ");
        }
        if (numlin) {
           indent -= numprog ? 6 : 3;
           strcpy(linull, numprog ? "      " : "   ");
           if (indent < 0)
              indent = 0;
        } else
           lino[0] = EOS;
        if (chgf) {
           expert = FALSE;         /* Cannot use change bars with expert */
           indent -= 2;
           if (indent < 0)
              indent = 0;
        }
        inds[0] = EOS;
        rhead[0] = EOS;
        for (i = 0; i < indent; i++)
           strcat(inds, " ");

        while ((!eof && getl()) ?
                 TRUE : (tbfr[0] = EOS, eof = TRUE, (lpu != 0 && pagelen))) {
           if (lpu == 0)
              pgtop();
           if (numlin) {
              if (eof || (tbfr[0] == EOS && !numprog))
                 strcpy(lino, linull);
              else
                 sprintf(lino, numprog ? "%4d: " : "%2d ", numprog ?
                         aline : ++lcount);
           }
           if (runhead && !rhead[0]) {
              if (numlin)
                 strcpy(rhead, linull);
              strcat(rhead, tbfr);
           }
           if (just && !expert)
              justify();
           if (!pcount)
              printf("%s%s%s%s\n", inds, lino, isbar(), tbfr);
           chkeop();
           if (dblspc) {
              if (!pcount)
                 printf("\n");
              chkeop();
           }
        }
        if (lpu && pagelen)
           pgbot();
        if (pcount)
           fprintf(stderr, "%d pages.\n", pageno);
        if (chgf)
           fclose(chgf);

        return 0;
}

/*  Check for end of page and perform end of page processing if there.  */

static void chkeop()
{
        if (pagelen && ++lpu >= pagelen) {
           pageno++;
           if (numpag && !pcount) {
              if (bookmode) {
                 printf("+PAGE %d", pageno);
              } else {
                 if (ofpages)
                    printf("\n%s%s-%d of %d-\n", inds, pnos, pageno, ofpages);
                 else
                    printf("\n%s%s-%d-\n", inds, pnos, pageno);
              }
           }
           pgbot();
           lpu = 0;
        }
}

/*  Perform top of page processing.  */

static void pgtop()
{
        int i;

        if (!bookmode) {
           for (i = 0; i < topmarg; i++) {
              if (!pcount)
                 printf("\n");
              lpu++;
           }
           if (rhead[0]) {
              if (!pcount)
                 printf("%s%s\n\n",inds,rhead);
              lpu += 2;
           }
        }
}

/*  Perform bottom of page processing.  */

static void pgbot()
{
        if (!pcount && !bookmode)
           printf(FORMFEED);
        lcount = 0;
}

/*  Test if character is an end of sentence delimiter.  */

static int issent(c)
  char c;
{
        return c == '.' || c == '!' || c == '?';
}

/*  Obtain next line, maintaining look ahead for justification control.  */

static int getl()
{
        static int first = TRUE,
                   pasteof = FALSE,
                   ateof = FALSE,
                   ffmode = FALSE;
        static char ffbuf[MAXIN];

        char *cp;

        if (ffmode) {
           if (lpu == 0) {
              ffmode = FALSE;
              if (*ffbuf) {
                 strcpy(tbfr, ffbuf);
                 return TRUE;
              }
           } else {
              *tbfr = EOS;
              return TRUE;
           }
        }

        if (pasteof)
           return FALSE;

        if (first) {
           first = FALSE;
           if (getexp(cbfr2)) {
              nbfr = cbfr2;
              tbfr = cbfr1;
           } else {
              pasteof = TRUE;
              return FALSE;
           }
        }
        cp = tbfr;
        tbfr = nbfr;
        nbfr = cp;
        if (ateof) {
           pasteof = TRUE;
           return FALSE;
        }
        if (!getexp(nbfr)) {
           ateof = TRUE;
           nbfr[0] = EOS;
        }
        aline++;
        if (formfeed && (*tbfr == FFCMD)) {
           strcpy(ffbuf, tbfr+1);
           *tbfr = EOS;
           if (lpu == 0) {
              strcpy(tbfr, ffbuf);
              return TRUE;
           }
           ffmode = TRUE;
        }
        return TRUE;
}

/*  Determine eligibility for inter-line filling  */

static int exelig(cp, cksent)
  char *cp;
  int cksent;
{
        int l;
        char *sp, c;

        l = strlen(cp);
        if (l == 0 || isspace(cp[0]))
           return FALSE;

        if (cksent && (!strncmp(cp, SENT1, strlen(SENT1)) ||
            ((l >= strlen(SENT2)) &&
              !strncmp(cp + l - strlen(SENT2), SENT2, strlen(SENT2)))))
           return FALSE;

        for (sp = cp; (c = *sp++) != EOS;) {
           if (issent(c) && isspace(*sp))
              sp++;
           else if (isspace(c) && ((c = *sp) != EOS) && isspace(c))
              return FALSE;
        }
        return TRUE;
}

/*  Determine if a character is "leading punctuation"  */

static int islp(c)
  char c;
{
        return (c == ' ' || (ispunct(c) &&
                (c != '\'' && c != '"' && c != '-')));
}

/*  Scan, recognise, and save prefix, returning its length  */

static int savepre(ibuf, obuf)
  char *ibuf, *obuf;
{
        char *cp, c, *cpt;
        int possbul;

        *obuf = EOS;
        possbul = FALSE;
        cp = ibuf;
        while (((c = *cp++) != EOS) && islp(c)) ;
        if (!c)
           return 0;
        cpt = cp;                 /* First possible position */

        /*  Suck up rest of possible hanging indent bullet  */

        while (c) {
           if (isalnum(c)) {
              while (((c = *cp++) != EOS) && isalnum(c)) ;
              if (issent(c)) {
                 c = *cp++;
                 if (c == ' ') {
                    c = *cp++;
                    if (c == ' ') {
                       possbul = TRUE;
                       break;
                    }
                 } else
                    continue;
              } else if (ispunct(c)) {
                 c = *cp++;
                 if (c == ' ') {
                    possbul = TRUE;
                    break;
                 } else
                    continue;
              } else
                 break;
           } else
              break;
        }

        /* If there's an extra space, consider this as a bullet and suck
           up all remaining spaces before the next text. */

        if (possbul && ((c = *cp++) != EOS) && c == ' ') {
           while (((c = *cp++)  != EOS) && c == ' ') ;
           cpt = cp;
        }

        cpt--;
        for (cp = ibuf; cp < cpt;)
           *obuf++ = *cp++;
        *obuf = EOS;
        return cpt - ibuf;
}

/*  Obtain input, performing requested text modification  */

static int getexp(buf)
  char *buf;
{
        static int lsaved = FALSE,    /* Line saved flag */
                   presaved = FALSE,  /* Prefix saved flag */
                   hiteof = FALSE,    /* End of file encountered */
                   indent2 = 0;       /* Indentation of second line */
        static char bline[MAXIN],       /* Saved line buffer */
                    prebuf[MAXIN];      /* Prefix buffer */

        int i, j, lo, ld, loi;
        char c;
        struct wd *w;

        if (!expert)
           return getins(buf);

        while (TRUE) {             /* Main state machine cycle */

           if (whn) {              /* Saved words remain */
              if (presaved) {
                 strcpy(buf, prebuf);
                 presaved = FALSE;
              } else {
                 for (i = 0; i < indent2; i++)
                    buf[i] = ' ';
                 buf[indent2] = EOS;
              }
              loi = lo = strlen(buf);
              ld = 0;
              w = whn;
              while (w && (ld == 0 || ((w->wlen + ld + lo) <= jlen))) {
                 strcpy(buf + lo, ld ? ((ld == 1) ? " " : "  ") : "");
                 lo += ld;
                 strcpy(buf + lo, w->wdt);
                 lo += w->wlen;
                 c = w->wdelim;
                 ld = (c == '.' || c == '?' || c == '!') ? 2 : 1;

                 whn = w->wdn;
                 free(w);
                 w = whn;
              }
              if (just)
                 i = jline(buf + loi, jlen - loi);
              return TRUE;
           }

           if (hiteof)
              return FALSE;

           /*  No previously saved data.  Enter state machine to process
               new lines.  */

           /*  First line of paragraph  */

           if (!lsaved) {
              if (!getins(bline))
                 return FALSE;
           }
           lsaved = FALSE;
           if (analyse(bline, buf)) {
              if (strlen(buf) == 0) {   /* Void forced line.  Just a break */
                 continue;
              }
              return TRUE;
           }
           i = savepre(bline, prebuf);
           presaved = TRUE;
           if (!exelig(bline + i, FALSE)) {
              presaved = FALSE;
              strcpy(buf, bline);
              return TRUE;
           }
           wordify(bline + i);    /* Wordify after prefix */

           /*  Second line of paragraph  */

           if (!getins(bline)) {
              hiteof = TRUE;
              continue;
           }
           i = strlen(bline);      /* Bypass indentation and save it */
           for (indent2 = 0; indent2 < i; indent2++) {
              if (bline[indent2] != ' ')
                 break;
           }
           if ((expert != 1 || indent2 == 0) &&
               exelig(bline + indent2, TRUE)) {
              wordify(bline + indent2);
           } else {
              lsaved = TRUE;
              continue;
           }

           /*  Lines 3 through N of a paragraph  */

           while (TRUE) {
              if (!getins(bline)) {
                 hiteof = TRUE;
                 break;
              }

              for (j = 0; j < indent2; j++)
                 if (bline[j] != ' ')
                    break;

              if (j != indent2 || bline[j] == ' ')
                 break;

              if (exelig(bline + indent2, TRUE)) {
                 i = strlen(bline);
                 for (j = 0; indent2 < i; indent2++) {
                    if (bline[indent2] != ' ')
                       break;
                 }
                 wordify(bline);
              } else
                 break;
           }
           lsaved = TRUE;
        }
}

/*  Obtain input from current source.  If no include file is active,
    we just read from standard input.  If an include file is being
    read, return lines from it until the end of file is encountered.
    Transparent commands are processed at this level.  */

static int getins(buf)
  char *buf;
{
        struct isource *s;
        FILE *fp;
        char c, *cp;
        char wuf[MAXIN];
        int l;

        while (TRUE) {
           if (istack) {
              if (!fgets(buf, MAXIN, istack->isfp)) {
                 s = istack;
                 istack = s->isnext;
                 free(s);
                 continue;
              }
              buf[strlen(buf) - 1] = EOS;
           } else {
              if (!fgets(buf, MAXIN, stdin))
                 return FALSE;
              buf[strlen(buf) - 1] = EOS;
           }
           detab(buf, wuf, SYSTABS);
           strcpy(buf, wuf);       /* Decompress text if compressed */

           l = strlen(buf);        /* Remove trailing blanks */
           while (l && (buf[--l] == ' '))
              buf[l] = EOS;

           /* Process include commands transparently */

           if (!strncmp(buf, INCLUDE, strlen(INCLUDE)) && strlen(buf + 2)) {
              if ((fp = fopen(buf + 2, "r")) == NULL) {
                 fprintf(stderr, "\nCannot open include file %s", buf + 2);
                 exit(1);
              }
              if ((s = (struct isource *)
                    malloc(sizeof(struct isource))) == NULL) {
                 fprintf(stderr, "\nIncludes nested too deeply.  Blew on %s",
                    buf + 2);
                 exit(1);
              }
              s->isnext = istack;
              s->isfp = fp;
              istack = s;
              continue;
           }

           /*  Check for embedded documentation sentinels and process
               accordingly.  */

           if (expert && !strcmp(buf, EMBDOC1)) {
              embact = embprog = TRUE;
              continue;
           }
           if (embprog) {
              if (!strcmp(buf, EMBDOC2)) {
                 embact = FALSE;
                 continue;
              }
              if (!embact)
                 continue;
           }

           /* Check for change justification column command and process
              if encountered. */

           if (!strncmp(buf, JCOL1, strlen(JCOL1))) {
              cp = buf + strlen(JCOL1);
              while ((c = *cp++) == ' ') ;
              if (c == JCOL2 && *cp == EOS) {
                 jlen = cp - buf;
                 continue;
              }
           }
           break;
        }
        return TRUE;
}

/*  Analyse a line and perform any special handling  */

static int analyse(tbfr, abfr)
  char *tbfr, *abfr;
{
        int i, l, rsent, lsent;
        char *lbuf;

        l = strlen(lbuf = tbfr);
        if ((lsent = !strncmp(tbfr, SENT1, strlen(SENT1))) != 0) {
           lbuf += strlen(SENT1);
           l -= strlen(SENT1);
        }
        if ((rsent = (((l >= strlen(SENT2)))) &&
                !strncmp(lbuf + l - strlen(SENT2), SENT2, strlen(SENT2))) != 0)
           l -= strlen(SENT2);
        else if (bookmode && tbfr[0] == '+')
           rsent = TRUE;

        if (l == 0 && (lsent || rsent)) {
           *abfr = EOS;
           return TRUE;
        }

        if (lsent) {
           /*  Centre line (RSENT present) or slam to right (RSENT absent) */
           lbuf[l] = EOS;
           *abfr = EOS;
           if ((jlen - l) > 0) {
              for (i = 0; i < (jlen - l) / (rsent ? 2 : 1);)
                 abfr[i++] = ' ';
              abfr[i] = EOS;
           }
           strcat(abfr, lbuf);
           return TRUE;
        } else if (rsent) {
           /*  Slam line to left (actually just breaks justification) */
           lbuf[l] = EOS;
           strcpy(abfr, lbuf);
           return TRUE;
        }
        return FALSE;
}

/*  Perform justification on a line, given a pointer to the line buffer
    and the number of columns it is to be justified to.  */

static int jline(tbfr, jlen)
  char *tbfr;
  int jlen;
{
        char c;
        char *cp;
        int l, d, n, od;
        static int parity = FALSE;

        l = strlen(tbfr);
        if (jlen <= 0)
           return -5;
        if (l == 0 || isspace(tbfr[0]))
           return -1;
        if (l < (jlen * jperc) / 100)
           return -2;
        if ((d = jlen - l) < 0)
           return -3;
        parity = !parity;
        while ((od = d) != 0) {
           if (parity) {
              for (n = l, cp = tbfr; n--, c = *cp++;) {
                 if (isspace(c) && (!isspace(*cp))) {
                    memcpy(cp + 1, cp, n + 1);
                    *cp++ = ' ';
                    l++;
                    if (--d <= 0)
                       return 1;
                 }
              }
              if (d == od)
                 return -6;
           } else {
              n = 0;
              cp = tbfr + l - 1;
              while (cp > tbfr) {
                 n++;
                 c = *cp--;
                 if (isspace(c) && (!isspace(*cp))) {
                    memcpy(cp + 2, cp + 1, n + 1);
                    l++;
                    n++;
                    if (--d <= 0)
                       return 1;
                 }
              }
              if (d == od)
                 return -6;
           }
        }
        return 0;
}

/*  Justify returns a code which explains what it did and why:

        0 = Line already correct length
        1 = Line justified normally

       -1 = Line has leading space, or next line has leading space
       -2 = Line too short, would look ridiculous if justified
       -3 = Line already too long
       -4 = Line contains significant embedded spaces
       -5 = Justification length zero or negative
       -6 = Line contains only one word
*/

static int justify()
{
        char c;
        char *cp;
        int l;

        l = strlen(tbfr);
        if (l == 0 || isspace(tbfr[0]) ||
            isspace(nbfr[0]) || (nbfr[0] == EOS))
           return -1;
        while (l > 0 && tbfr[l - 1] == ' ')
           tbfr[--l] = EOS;
        for (cp = tbfr; (c = *cp++) != EOS;) {
           if (issent(c) && isspace(*cp))
              cp++;
           else if (isspace(c) && ((c = *cp) != EOS) && isspace(c))
              return -4;
        }
        return jline(tbfr, jlen);
}

/*  Determine if this line is changed from the DIFF file, and return a
    change bar string if so.  If change bars are not desired, just return
    a null string.  */

static char *isbar()
{
        char cfs[132];
        char minus, comm, eol;
        int from, to, nsc;
        static int crangel = -1, crangeh = -1;
        static int mflag = FALSE, mrange = 0;

        while (chgf) {
           if (eof)
              return "  ";
           if (aline < crangel) {
              if (mflag && (aline >= mrange)) {
                 mflag = FALSE;
                 return (aline == mrange) ? "- " : "  ";
              } else
                 return "  ";
           }
           if (aline <= crangeh)
              return "| ";
           while (TRUE) {
              if (fgets(cfs, 132, chgf)) {
                 nsc = sscanf(cfs, "%c%d%c%d%c", &minus,
                              &from, &comm, &to, &eol);
                 if (nsc == 5 && minus == '-' &&
                     comm == ',' && eol == '\n') {
                    crangel = from;
                    crangeh = to;
                    break;
                 }
                 if (nsc == 3 && minus == '-' && comm == '\n') {
                    mrange = from + 1;
                    mflag = TRUE;
                 }
                 continue;
              } else {
                 crangel = INT_MAX;
                 return "  ";
              }
           }
        }
        return "";
}

/*  Allocate a buffer, detecting out of memory and gracefully handling it.  */

static char *alloc(nb)
  int nb;
{
        char *cp;

        if ((cp = malloc(nb)) == NULL) {
           fprintf(stderr, "\nBoom!!!  Memory capacity exceeded.");
           exit(1);
        }
        return cp;
}

/*  Scan an input line and decompose into words.  */

static void wordify(cp)
  char *cp;
{
        char c;
        int i;
        struct wd *w;
        char wbuf[MAXIN];

        while (*cp) {
           while (((c = *cp++) != EOS) && (c == ' ')) ;
           if (c == EOS) {
              cp--;
              break;
           }

           i = 0;
           wbuf[i++] = c;

           while (((c = *cp++) != EOS) && (c != ' ')) {
              wbuf[i++] = c;
           }
           if (c == EOS)
              cp--;
           wbuf[i] = EOS;
           w = (struct wd *) alloc(sizeof(struct wd) + i + 1);
           w->wlen = i;
           w->wdelim = wbuf[i - 1];
           strcpy(w->wdt, wbuf);

           if (!whn) {
              whn = whp = w;
              w->wdp = w->wdn = NULL;
           } else {
              whp->wdn = w;
              w->wdp = whp->wdn;
              w->wdn = NULL;
              whp = w;
           }

        }
}

/*  Remove tabs from the input, allowing input from editors which tab
    their output.  The argument NC specifies the default tabs of the
    system.  */

static void detab(ibuf, obuf, nc)
  char *ibuf, *obuf;
  int nc;
{
        char c;
        int col = 0;

        while ((c = *ibuf++) != EOS) {
           if (c == '\t') {
              *obuf++ = ' ';
              col++;
              while (col % nc)
                 *obuf++ = ' ', col++;
#ifdef DECTL
           } else if (c < ' ') {
              continue;
#endif
           } else
              *obuf++ = c, col++;
        }
        *obuf++ = c;
}
